// src/services/PhonePeService.js
// DEMO VERSION - Bypasses PhonePe, directly creates package purchase
// यह demo version है - PhonePe को skip करके सीधे database में entry करेगा

import axios from "axios";
import { URL } from "../config";

/**
 * Generate unique transaction ID
 */
const generateTransactionId = () => {
  const timestamp = Date.now();
  const random = Math.random().toString(36).substr(2, 9).toUpperCase();
  return `DEMO${timestamp}${random}`;
};

/**
 * Get User ID from user object
 */
const getUserId = (user) => {
  if (!user) return null;

  return (
    user.id ||
    user.userId ||
    user.sellerId ||
    user.buyerId ||
    user.fkBuyerIdSellerId ||
    user.fkbuyerId ||
    user.user_id ||
    null
  );
};

/**
 * DEMO Payment - Directly creates package purchase without PhonePe
 * डेमो पेमेंट - PhonePe के बिना सीधे package purchase बनाएगा
 */
export const initiatePhonePePayment = async (paymentDetails) => {
  try {
    console.log("=== DEMO Payment Started ===");
    console.log("Payment Details:", paymentDetails);

    // Extract data
    const {
      amount,
      userId: providedUserId,
      userName,
      userPhone,
      userEmail,
      planName,
      packageId,
      packageDuration,
      propertyVisit,
      propertyAddToCart,
    } = paymentDetails;

    // Get user ID
    let userId = providedUserId;
    let user = null;

    if (!userId) {
      const storedUser = localStorage.getItem("user");
      if (storedUser) {
        user = JSON.parse(storedUser);
        userId = getUserId(user);
      }
    }

    // Validate
    if (!userId) {
      throw new Error("User ID is required. Please log in again.");
    }

    if (!packageId) {
      throw new Error("Package ID is required");
    }

    console.log("User ID:", userId, "Package:", packageId);

    // Generate demo transaction ID
    const transactionId = generateTransactionId();
    console.log("Demo Transaction ID:", transactionId);

    // Save transaction details
    saveTransactionDetails({
      transactionId: transactionId,
      amount: amount || 0,
      status: "DEMO_COMPLETED",
      planName: planName,
      packageId: packageId,
      packageDuration: packageDuration,
      propertyVisit: propertyVisit,
      propertyAddToCart: propertyAddToCart,
      userId: userId,
      userName: userName || user?.name || "",
      userPhone: userPhone || user?.phone || "",
      userEmail: userEmail || user?.email || "",
      timestamp: new Date().toISOString(),
    });

    // Simulate processing delay
    await new Promise((resolve) => setTimeout(resolve, 1000));

    // Return success with demo callback URL
    console.log("✅ Demo payment successful");

    return {
      success: true,
      isDemoMode: true,
      redirectUrl: `${window.location.origin}/payment/callback?orderId=${transactionId}`,
      transactionId: transactionId,
    };
  } catch (error) {
    console.error("=== Demo Payment Error ===");
    console.error("Error:", error);

    return {
      success: false,
      error: error.message || "Demo payment failed",
    };
  }
};

/**
 * DEMO Payment Status - Always returns success
 * डेमो स्टेटस - हमेशा success return करेगा
 */
export const checkPaymentStatus = async (merchantTransactionId) => {
  try {
    console.log("=== DEMO Status Check ===");
    console.log("Transaction ID:", merchantTransactionId);

    // Get transaction from localStorage
    const transactions = JSON.parse(
      localStorage.getItem("transactions") || "[]",
    );
    const transaction = transactions.find(
      (t) => t.transactionId === merchantTransactionId,
    );

    if (!transaction) {
      console.warn("Transaction not found, creating demo success status");
    }

    // Always return success for demo
    const demoAmount = transaction?.amount || 100;

    console.log("✅ Demo payment verified as successful");

    return {
      success: true,
      status: "COMPLETED",
      amount: demoAmount,
      transactionId: merchantTransactionId,
      merchantTransactionId: merchantTransactionId,
      isDemoMode: true,
      responseCode: "DEMO_SUCCESS",
    };
  } catch (error) {
    console.error("Demo status check error:", error);

    // Even on error, return success for demo
    return {
      success: true,
      status: "COMPLETED",
      amount: 100,
      transactionId: merchantTransactionId,
      isDemoMode: true,
      responseCode: "DEMO_SUCCESS",
    };
  }
};

/**
 * Save Transaction Details
 */
const saveTransactionDetails = (transactionData) => {
  try {
    const transactions = JSON.parse(
      localStorage.getItem("transactions") || "[]",
    );
    transactions.push(transactionData);
    localStorage.setItem("transactions", JSON.stringify(transactions));
    console.log("Transaction saved:", transactionData.transactionId);
  } catch (error) {
    console.error("Error saving transaction:", error);
  }
};

const processingTransactions = new Map();

/**
 * Verify Payment and Create Package Purchase
 * यह function database में entry करेगा
 */
export const verifyPaymentAndCreatePurchase = async (transactionId) => {
  try {
    console.log("=== Creating Package Purchase (DEMO) ===");
    console.log("Transaction ID:", transactionId);

    if (processingTransactions.has(transactionId)) {
      console.log("Transaction already being processed:", transactionId);
      return {
        success: false,
        error: "Transaction is already being processed",
      };
    }

    processingTransactions.set(transactionId, true);

    try {
      const transactions = JSON.parse(
        localStorage.getItem("transactions") || "[]",
      );
      const transaction = transactions.find(
        (t) => t.transactionId === transactionId,
      );

      if (!transaction) {
        console.warn("Transaction not found in localStorage");
      }

      const userStr = localStorage.getItem("user");
      if (!userStr) {
        throw new Error("User not found in localStorage");
      }

      const user = JSON.parse(userStr);
      const userId = getUserId(user);

      if (!userId) {
        throw new Error("User ID not found");
      }

      const usertypeId =
        user.userTypeId ||
        user.fkusertypeId ||
        user.fkuserId ||
        user.usertypeId ||
        1;

      console.log("User ID:", userId, "Usertype ID:", usertypeId);

      const purchaseData = {
        transactionId: transactionId,
        packageId: transaction?.packageId || 1,
        userId: userId,
        amount: transaction?.amount || 0,
        packageDuration: transaction?.packageDuration || 30,
        propertyVisit: transaction?.propertyVisit || 10,
        propertyAddToCart: transaction?.propertyAddToCart || 5,
        usertypeId: usertypeId,
        isDemoMode: true,
      };

      console.log("Creating package purchase with data:", purchaseData);

      const response = await axios.post(
        `${URL}/package-purchases/verify-payment`,
        purchaseData,
        {
          headers: {
            "Content-Type": "application/json",
          },
        },
      );

      console.log("Backend Response:", response.data);

      if (
        response.data &&
        (response.data.packagePurchaseId || response.data.id)
      ) {
        const currentDate = new Date();
        const packageEndDate = new Date();
        packageEndDate.setDate(
          currentDate.getDate() + parseInt(transaction?.packageDuration || 30),
        );

        user.isPrime = true;
        user.membershipExpiry = packageEndDate.toISOString();
        user.activePackage = {
          packageId: transaction?.packageId || 1,
          packageName: transaction?.planName || "Demo Package",
          packagePurchaseId:
            response.data.packagePurchaseId || response.data.id,
          startDate: currentDate.toISOString(),
          endDate: packageEndDate.toISOString(),
          transactionId: transactionId,
          isDemoMode: true,
        };

        localStorage.setItem("user", JSON.stringify(user));
        console.log("✅ User updated with demo membership");

        return {
          success: true,
          data: response.data,
          message: "Demo package purchase created successfully!",
          isDemoMode: true,
        };
      } else {
        throw new Error("Invalid response from server");
      }
    } finally {
      processingTransactions.delete(transactionId);
    }
  } catch (error) {
    console.error("=== Package Purchase Error ===");
    console.error("Error:", error);

    processingTransactions.delete(transactionId);

    if (error.response) {
      console.error("Response:", error.response.data);
      console.error("Status:", error.response.status);
    }

    return {
      success: false,
      error:
        error.response?.data?.message ||
        error.message ||
        "Failed to create package purchase",
    };
  }
};
export default {
  initiatePhonePePayment,
  checkPaymentStatus,
  verifyPaymentAndCreatePurchase,
};
