import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { motion } from "framer-motion";
import axios from "axios";
import {
  Box,
  Grid,
  Card,
  CardMedia,
  CardContent,
  Typography,
  Button,
  Chip,
  Stack,
  Rating,
  Container,
  Avatar,
  IconButton,
  Tooltip,
  CircularProgress,
  Alert,
  LinearProgress,
} from "@mui/material";
import {
  LocationOn,
  Bed,
  Bathtub,
  SquareFoot,
  Balcony,
  Apartment,
  Verified,
  Phone,
  Info,
  History,
  Visibility,
  Delete,
} from "@mui/icons-material";
import { Favorite as HeartOutline } from "@mui/icons-material";
import { message, Modal } from "antd";
import { ExclamationCircleOutlined } from "@ant-design/icons";
import img1 from "../../assets/PostIMG/PostImg_3.jpeg";
import img2 from "../../assets/PostIMG/PostImg_4.jpeg";
import { URL } from "../../config";

const fallbackImages = [img1, img2];

// Function to format price in Lakh and CR format
const formatPrice = (price) => {
  if (!price) return "N/A";

  const crore = 10000000;
  const lakh = 100000;

  if (price >= crore) {
    return `${(price / crore).toFixed(2)} CR`;
  } else if (price >= lakh) {
    return `${(price / lakh).toFixed(2)} Lakh`;
  } else {
    return `₹${price.toLocaleString()}`;
  }
};

// Function to format date
const formatDate = (dateString) => {
  if (!dateString) return "Recently";

  const date = new Date(dateString);
  const now = new Date();
  const diffTime = Math.abs(now - date);
  const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

  if (diffDays === 0) return "Today";
  if (diffDays === 1) return "Yesterday";
  if (diffDays < 7) return `${diffDays} days ago`;
  if (diffDays < 30)
    return `${Math.floor(diffDays / 7)} week${Math.floor(diffDays / 7) > 1 ? "s" : ""} ago`;
  if (diffDays < 365)
    return `${Math.floor(diffDays / 30)} month${Math.floor(diffDays / 30) > 1 ? "s" : ""} ago`;
  return `${Math.floor(diffDays / 365)} year${Math.floor(diffDays / 365) > 1 ? "s" : ""} ago`;
};

const ViewPropertyPage = () => {
  const [viewedList, setViewedList] = useState([]);
  const [loading, setLoading] = useState(true);
  const [user, setUser] = useState(null);
  const [activePackage, setActivePackage] = useState(null);
  const [viewCount, setViewCount] = useState(0);
  const navigate = useNavigate();

  // Get user from localStorage
  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);

      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();

        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }
  }, []);

  // Fetch viewed properties
  useEffect(() => {
    const fetchViewedProperties = async () => {
      if (!user) {
        setLoading(false);
        return;
      }

      try {
        setLoading(true);
        let url = "";

        if (user?.fkusertypeId === 1 && user?.buyerId) {
          url = `${URL}/log-properties/buyer/${user.buyerId}`;
        } else if (user?.fkusertypeId !== 1 && user?.sellerId) {
          url = `${URL}/log-properties/seller/${user.sellerId}`;
        }

        const response = await axios.get(url);

        // Sort by most recent first
        const sortedData = response.data.sort(
          (a, b) => new Date(b.createdAt) - new Date(a.createdAt),
        );

        setViewedList(sortedData);
        setViewCount(sortedData.length);
      } catch (error) {
        console.error("Error fetching viewed properties:", error);
        message.error("Failed to load viewed properties");
      } finally {
        setLoading(false);
      }
    };

    fetchViewedProperties();
  }, [user]);

  const handleRemoveFromHistory = async (logPropertyId, propertyTitle) => {
    Modal.confirm({
      title: "Remove from History?",
      icon: <ExclamationCircleOutlined />,
      content: `Are you sure you want to remove "${propertyTitle}" from your viewing history?`,
      okText: "Yes, Remove",
      cancelText: "Cancel",
      okButtonProps: { danger: true },
      onOk: async () => {
        try {
          await axios.delete(`${URL}/log-properties/${logPropertyId}`);
          setViewedList(
            viewedList.filter((item) => item.logPropertyId !== logPropertyId),
          );
          setViewCount((prev) => prev - 1);
          message.success("Property removed from history");
        } catch (error) {
          console.error("Error removing property from history:", error);
          message.error("Failed to remove property from history");
        }
      },
    });
  };

  const handleAddToCart = async (propertyId, propertyTitle) => {
    if (!user) {
      message.warning("Please login to save properties");
      navigate("/login");
      return;
    }

    try {
      // Check if already in cart
      let cartUrl = "";
      if (user.buyerId) {
        cartUrl = `${URL}/add-to-carts/buyer/${user.buyerId}`;
      } else if (user.sellerId) {
        cartUrl = `${URL}/add-to-carts/seller/${user.sellerId}`;
      }

      const cartResponse = await axios.get(cartUrl);
      const isInCart = cartResponse.data.some(
        (item) => item.fkPostpropertyId === propertyId,
      );

      if (isInCart) {
        message.info("Property is already in your saved list");
        return;
      }

      // Add to cart
      const cartData = {
        fkPostpropertyId: propertyId,
        fkBuyerId: user.buyerId || null,
        fkSellerId: user.sellerId || null,
      };

      await axios.post(`${URL}/add-to-carts`, cartData);
      message.success(`"${propertyTitle}" added to saved properties`);
    } catch (error) {
      console.error("Error adding to cart:", error);
      message.error("Failed to add property to saved list");
    }
  };

  const handleCardClick = (propertyId) => {
    localStorage.setItem("id", propertyId);
    navigate(`/Propertyinfo/${propertyId}`);
  };

  const handleViewDetails = (propertyId) => {
    localStorage.setItem("id", propertyId);
    navigate(`/Propertyinfo/${propertyId}`);
  };

  // Calculate view usage percentage
  const getViewUsagePercentage = () => {
    if (!activePackage) return 0;
    const limit = activePackage.propertyVisit || 0;
    if (limit === 0) return 0;
    return Math.min((viewCount / limit) * 100, 100);
  };

  if (!user) {
    return (
      <Box
        sx={{
          minHeight: "100vh",
          display: "flex",
          alignItems: "center",
          justifyContent: "center",
          backgroundColor: "#f8f9fa",
        }}
      >
        <Box sx={{ textAlign: "center" }}>
          <Typography variant="h4" sx={{ mb: 2, fontFamily: "Poppins" }}>
            Please Login
          </Typography>
          <Typography
            variant="body1"
            sx={{ mb: 3, color: "#666", fontFamily: "Poppins" }}
          >
            Login to view your property history
          </Typography>
          <Button
            variant="contained"
            onClick={() => navigate("/login")}
            sx={{
              backgroundColor: "#1976d2",
              fontFamily: "Poppins",
              px: 4,
              py: 1.5,
            }}
          >
            Login
          </Button>
        </Box>
      </Box>
    );
  }

  return (
    <div style={{ backgroundColor: "#f8f9fa", minHeight: "100vh" }}>
      <Container
        maxWidth="lg"
        sx={{
          py: { xs: 5, md: 8 },
          mt: { xs: 7, md: 8 },
        }}
      >
        {/* Header Section */}
        <Box
          sx={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            mb: 3,
            flexWrap: "wrap",
            gap: 2,
          }}
        >
          <Box>
            <Typography
              variant="h2"
              sx={{
                fontSize: { xs: "28px", md: "42px" },
                fontWeight: 700,
                mb: 1,
                background: "linear-gradient(135deg, #1976d2, #42a5f5)",
                WebkitBackgroundClip: "text",
                WebkitTextFillColor: "transparent",
                fontFamily: "Poppins",
              }}
            >
              Property View History
            </Typography>
            <Typography
              variant="body1"
              sx={{
                fontSize: "16px",
                color: "#666",
                fontFamily: "Poppins",
              }}
            >
              {viewedList.length}{" "}
              {viewedList.length === 1 ? "property" : "properties"} you have
              viewed
            </Typography>
          </Box>

          <Button
            variant="contained"
            onClick={() => navigate("/property-listing")}
            sx={{
              backgroundColor: "#1976d2",
              color: "white",
              px: 4,
              py: 1.5,
              borderRadius: "8px",
              fontSize: "15px",
              fontWeight: 600,
              boxShadow: "0 4px 12px rgba(25,118,210,0.3)",
              transition: "all 0.3s ease",
              "&:hover": {
                backgroundColor: "#1565c0",
                transform: "translateY(-2px)",
                boxShadow: "0 6px 20px rgba(25,118,210,0.4)",
              },
              fontFamily: "Poppins",
            }}
          >
            Explore Properties ➜
          </Button>
        </Box>

        {/* Package Status Card - Show if user has package */}
        {activePackage && (
          <Card
            sx={{
              mb: 4,
              borderRadius: 3,
              overflow: "hidden",
              background:
                "linear-gradient(135deg, rgba(25,118,210,0.05) 0%, rgba(66,165,245,0.05) 100%)",
              border: "1px solid rgba(25,118,210,0.1)",
            }}
          >
            <CardContent sx={{ p: 3 }}>
              <Stack direction="row" spacing={2} alignItems="center" mb={2}>
                <Avatar sx={{ bgcolor: "#1976d2", width: 48, height: 48 }}>
                  <History />
                </Avatar>
                <Box>
                  <Typography
                    variant="h6"
                    sx={{ fontWeight: 700, fontFamily: "Poppins" }}
                  >
                    {activePackage.packageName}
                  </Typography>
                  <Typography
                    variant="body2"
                    sx={{ color: "#666", fontFamily: "Poppins" }}
                  >
                    Active Package
                  </Typography>
                </Box>
              </Stack>

              <Box sx={{ mb: 2 }}>
                <Stack direction="row" justifyContent="space-between" mb={1}>
                  <Typography
                    variant="body2"
                    sx={{ color: "#666", fontFamily: "Poppins" }}
                  >
                    Property Views
                  </Typography>
                  <Typography
                    variant="body2"
                    sx={{ fontWeight: 600, fontFamily: "Poppins" }}
                  >
                    {viewCount} / {activePackage.propertyVisit || 0} views used
                  </Typography>
                </Stack>

                <LinearProgress
                  variant="determinate"
                  value={getViewUsagePercentage()}
                  sx={{
                    height: 8,
                    borderRadius: 1,
                    backgroundColor: "rgba(25,118,210,0.1)",
                    "& .MuiLinearProgress-bar": {
                      backgroundColor:
                        getViewUsagePercentage() >= 100 ? "#f44336" : "#1976d2",
                      borderRadius: 1,
                    },
                  }}
                />

                <Typography
                  variant="caption"
                  sx={{
                    color: getViewUsagePercentage() >= 100 ? "#f44336" : "#666",
                    display: "block",
                    mt: 0.5,
                    fontFamily: "Poppins",
                  }}
                >
                  {getViewUsagePercentage() >= 100
                    ? "⚠️ You've reached your view limit! Upgrade to view more properties."
                    : `${Math.round(getViewUsagePercentage())}% of your view limit used`}
                </Typography>
              </Box>

              {getViewUsagePercentage() >= 100 && (
                <Alert
                  severity="error"
                  sx={{
                    mt: 2,
                    borderRadius: 2,
                  }}
                  action={
                    <Button
                      color="inherit"
                      size="small"
                      onClick={() => navigate("/membership")}
                    >
                      Upgrade
                    </Button>
                  }
                >
                  You've reached your view limit. Please upgrade your package to
                  view more properties.
                </Alert>
              )}

              {getViewUsagePercentage() >= 80 &&
                getViewUsagePercentage() < 100 && (
                  <Alert
                    severity="warning"
                    sx={{
                      mt: 2,
                      borderRadius: 2,
                    }}
                    action={
                      <Button
                        color="inherit"
                        size="small"
                        onClick={() => navigate("/membership")}
                      >
                        View Plans
                      </Button>
                    }
                  >
                    You're running out of views! Only{" "}
                    {(activePackage.propertyVisit || 0) - viewCount} views
                    remaining.
                  </Alert>
                )}
            </CardContent>
          </Card>
        )}

        {/* No Package Warning */}
        {!activePackage && viewedList.length > 0 && (
          <Alert
            severity="info"
            sx={{
              mb: 4,
              borderRadius: 2,
            }}
            icon={<Info />}
            action={
              <Button
                color="inherit"
                size="small"
                onClick={() => navigate("/membership")}
              >
                Get Package
              </Button>
            }
          >
            <Typography
              variant="body2"
              sx={{ fontFamily: "Poppins", fontWeight: 600 }}
            >
              Get a package to unlock unlimited property views
            </Typography>
            <Typography variant="caption" sx={{ fontFamily: "Poppins" }}>
              Purchase a package to view unlimited properties and access premium
              features.
            </Typography>
          </Alert>
        )}

        {/* Loading State */}
        {loading ? (
          <Box
            sx={{
              display: "flex",
              justifyContent: "center",
              alignItems: "center",
              minHeight: "400px",
            }}
          >
            <CircularProgress size={60} />
          </Box>
        ) : viewedList.length === 0 ? (
          // Empty State
          <Box
            sx={{
              textAlign: "center",
              py: 10,
              backgroundColor: "white",
              borderRadius: "16px",
              boxShadow: "0 4px 12px rgba(0,0,0,0.05)",
            }}
          >
            <History
              style={{ fontSize: 80, color: "#ccc", marginBottom: 16 }}
            />
            <Typography
              variant="h5"
              sx={{ mb: 2, color: "#666", fontFamily: "Poppins" }}
            >
              No Property View History Yet
            </Typography>
            <Typography
              variant="body1"
              sx={{ mb: 4, color: "#999", fontFamily: "Poppins" }}
            >
              Start exploring properties to build your viewing history
            </Typography>
            <Button
              variant="contained"
              size="large"
              onClick={() => navigate("/ ")}
              sx={{
                backgroundColor: "#1976d2",
                fontFamily: "Poppins",
                px: 4,
                py: 1.5,
              }}
            >
              Home
            </Button>
          </Box>
        ) : (
          // Property Cards Grid
          <Grid container spacing={4}>
            {viewedList.map((item, index) => {
              const property = item.postProperty;
              const imageSrc =
                property?.photos?.[0] ||
                property?.photo ||
                fallbackImages[index % fallbackImages.length];

              const rating = property?.rating || 4.2;

              return (
                <Grid
                  item
                  xs={12}
                  sm={6}
                  lg={4}
                  component={motion.div}
                  key={item.logPropertyId || index}
                  initial={{ opacity: 0, y: 50 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: false, amount: 0.3 }}
                  transition={{ duration: 0.6, delay: index * 0.15 }}
                >
                  <Card
                    onClick={() => handleCardClick(item.fkPostpropertyId)}
                    sx={{
                      borderRadius: "16px",
                      overflow: "hidden",
                      cursor: "pointer",
                      transition: "all 0.3s ease",
                      height: "100%",
                      display: "flex",
                      flexDirection: "column",
                      boxShadow: "0 8px 24px rgba(0,0,0,0.1)",
                      "&:hover": {
                        transform: "translateY(-8px)",
                        boxShadow: "0 12px 32px rgba(0,0,0,0.15)",
                      },
                    }}
                  >
                    {/* Image Section with Overlay */}
                    <Box sx={{ position: "relative" }}>
                      <CardMedia
                        component="img"
                        image={imageSrc}
                        alt={property?.Property_Title || "Property"}
                        onError={(e) => {
                          e.target.onerror = null;
                          e.target.src =
                            fallbackImages[index % fallbackImages.length];
                        }}
                        sx={{
                          height: { xs: 220, sm: 240, md: 260 },
                          width: "100%",
                          objectFit: "cover",
                        }}
                      />

                      {/* Gradient Overlay */}
                      <Box
                        sx={{
                          position: "absolute",
                          top: 0,
                          left: 0,
                          right: 0,
                          bottom: 0,
                          background:
                            "linear-gradient(to bottom, transparent 60%, rgba(0,0,0,0.7) 100%)",
                        }}
                      />

                      {/* Top Badges */}
                      <Stack
                        direction="row"
                        spacing={1}
                        sx={{
                          position: "absolute",
                          top: 12,
                          left: 12,
                          zIndex: 1,
                        }}
                      >
                        <Chip
                          icon={<Visibility sx={{ fontSize: 14 }} />}
                          label="Viewed"
                          size="small"
                          sx={{
                            bgcolor: "rgba(255,255,255,0.95)",
                            color: "#1976d2",
                            fontWeight: 600,
                            fontSize: "0.7rem",
                            borderRadius: "20px",
                            backdropFilter: "blur(10px)",
                          }}
                        />
                      </Stack>

                      <Stack
                        direction="row"
                        spacing={1}
                        sx={{
                          position: "absolute",
                          top: 12,
                          right: 12,
                          zIndex: 1,
                        }}
                      >
                        <Chip
                          label={property?.postType?.posttypeName || "For Sale"}
                          size="small"
                          sx={{
                            bgcolor: "rgba(255, 152, 0, 0.9)",
                            color: "#fff",
                            fontWeight: "bold",
                            fontSize: "0.7rem",
                            borderRadius: "20px",
                            backdropFilter: "blur(10px)",
                          }}
                        />
                      </Stack>

                      {/* View Date and Action Buttons */}
                      <Stack
                        direction="row"
                        spacing={1}
                        sx={{
                          position: "absolute",
                          bottom: 12,
                          left: 12,
                          right: 12,
                          zIndex: 1,
                          justifyContent: "space-between",
                        }}
                      >
                        <Chip
                          label={formatDate(item.createdAt)}
                          size="small"
                          sx={{
                            bgcolor: "rgba(0,0,0,0.6)",
                            color: "#fff",
                            fontWeight: 600,
                            fontSize: "0.7rem",
                            borderRadius: "20px",
                            backdropFilter: "blur(10px)",
                          }}
                        />
                        {/* 
                        <Stack direction="row" spacing={1}>
                          <Tooltip title="Add to saved">
                            <IconButton
                              size="small"
                              sx={{
                                bgcolor: "rgba(255,255,255,0.9)",
                                "&:hover": { bgcolor: "rgba(25,118,210,0.1)" },
                              }}
                              onClick={(e) => {
                                e.stopPropagation();
                                handleAddToCart(
                                  item.fkPostpropertyId,
                                  property?.Property_Title
                                );
                              }}
                            >
                              <HeartOutline sx={{ fontSize: 20, color: "#1976d2" }} />
                            </IconButton>
                          </Tooltip>

                          <Tooltip title="Remove from history">
                            <IconButton
                              size="small"
                              sx={{
                                bgcolor: "rgba(255,255,255,0.9)",
                                "&:hover": { bgcolor: "rgba(244,67,54,0.1)" },
                              }}
                              onClick={(e) => {
                                e.stopPropagation();
                                handleRemoveFromHistory(
                                  item.logPropertyId,
                                  property?.Property_Title
                                );
                              }}
                            >
                              <Delete sx={{ fontSize: 20, color: "#f44336" }} />
                            </IconButton>
                          </Tooltip>
                        </Stack> */}
                      </Stack>
                    </Box>

                    {/* Content Section */}
                    <CardContent
                      sx={{
                        p: 3,
                        flexGrow: 1,
                        display: "flex",
                        flexDirection: "column",
                        bgcolor: "white",
                      }}
                    >
                      {/* Title and Location */}
                      <Box mb={2}>
                        <Typography
                          variant="h6"
                          sx={{
                            fontSize: { xs: "1.1rem", md: "1.25rem" },
                            fontWeight: 700,
                            color: "#1a1a1a",
                            mb: 0.5,
                            lineHeight: 1.3,
                            fontFamily: "Poppins",
                          }}
                        >
                          {property?.Property_Title || "Beautiful Home"}
                        </Typography>

                        <Stack
                          direction="row"
                          alignItems="center"
                          spacing={0.5}
                          sx={{ mt: 0.5 }}
                        >
                          <LocationOn
                            sx={{
                              fontSize: 16,
                              color: "#1976d2",
                            }}
                          />
                          <Typography
                            variant="body2"
                            sx={{
                              color: "#666",
                              fontSize: "0.9rem",
                              fontFamily: "Poppins",
                            }}
                          >
                            {property?.sublocality ||
                              property?.fklocalityId ||
                              property?.fkcityId ||
                              "Location not available"}
                          </Typography>
                        </Stack>
                      </Box>

                      {/* Property Type Badge */}
                      <Stack
                        direction="row"
                        alignItems="center"
                        spacing={1}
                        mb={2}
                      >
                        <Chip
                          icon={<Apartment sx={{ fontSize: 14 }} />}
                          label={`${
                            property?.propertyType?.propertytypeName ||
                            "Property"
                          } • ${
                            property?.mainPropertyType?.mainprpoertyName ||
                            "Type"
                          }`}
                          size="small"
                          variant="outlined"
                          sx={{
                            borderColor: "#e0e0e0",
                            color: "#666",
                            fontSize: "0.75rem",
                            fontFamily: "Poppins",
                          }}
                        />
                      </Stack>

                      {/* Rating */}
                      <Stack
                        direction="row"
                        alignItems="center"
                        spacing={1}
                        mb={2}
                      >
                        <Rating
                          value={rating}
                          precision={0.5}
                          readOnly
                          size="small"
                          sx={{ color: "#ffc107" }}
                        />
                        <Typography
                          variant="body2"
                          sx={{
                            fontWeight: 600,
                            color: "#333",
                            fontSize: "0.9rem",
                          }}
                        >
                          {rating.toFixed(1)}
                        </Typography>
                      </Stack>

                      {/* Property Features */}
                      <Box
                        sx={{
                          display: "grid",
                          gridTemplateColumns: "repeat(2, 1fr)",
                          gap: 1.5,
                          mb: 2,
                        }}
                      >
                        {property?.noofbedrooms && (
                          <Stack
                            direction="row"
                            alignItems="center"
                            spacing={1}
                          >
                            <Box
                              sx={{
                                p: 0.5,
                                borderRadius: 1,
                                bgcolor: "#f5f5f5",
                                display: "flex",
                                alignItems: "center",
                                justifyContent: "center",
                              }}
                            >
                              <Bed sx={{ fontSize: 18, color: "#1976d2" }} />
                            </Box>
                            <Typography
                              variant="body2"
                              sx={{
                                color: "#555",
                                fontSize: "0.85rem",
                                fontFamily: "Poppins",
                              }}
                            >
                              {property.noofbedrooms} Beds
                            </Typography>
                          </Stack>
                        )}

                        {property?.noofbathrooms && (
                          <Stack
                            direction="row"
                            alignItems="center"
                            spacing={1}
                          >
                            <Box
                              sx={{
                                p: 0.5,
                                borderRadius: 1,
                                bgcolor: "#f5f5f5",
                                display: "flex",
                                alignItems: "center",
                                justifyContent: "center",
                              }}
                            >
                              <Bathtub
                                sx={{ fontSize: 18, color: "#1976d2" }}
                              />
                            </Box>
                            <Typography
                              variant="body2"
                              sx={{
                                color: "#555",
                                fontSize: "0.85rem",
                                fontFamily: "Poppins",
                              }}
                            >
                              {property.noofbathrooms} Baths
                            </Typography>
                          </Stack>
                        )}

                        {property?.builduparea && (
                          <Stack
                            direction="row"
                            alignItems="center"
                            spacing={1}
                          >
                            <Box
                              sx={{
                                p: 0.5,
                                borderRadius: 1,
                                bgcolor: "#f5f5f5",
                                display: "flex",
                                alignItems: "center",
                                justifyContent: "center",
                              }}
                            >
                              <SquareFoot
                                sx={{ fontSize: 18, color: "#1976d2" }}
                              />
                            </Box>
                            <Typography
                              variant="body2"
                              sx={{
                                color: "#555",
                                fontSize: "0.85rem",
                                fontFamily: "Poppins",
                              }}
                            >
                              {property.supperbuilduparea} sqft
                            </Typography>
                          </Stack>
                        )}

                        {property?.noofbalkanies && (
                          <Stack
                            direction="row"
                            alignItems="center"
                            spacing={1}
                          >
                            <Box
                              sx={{
                                p: 0.5,
                                borderRadius: 1,
                                bgcolor: "#f5f5f5",
                                display: "flex",
                                alignItems: "center",
                                justifyContent: "center",
                              }}
                            >
                              <Balcony
                                sx={{ fontSize: 18, color: "#1976d2" }}
                              />
                            </Box>
                            <Typography
                              variant="body2"
                              sx={{
                                color: "#555",
                                fontSize: "0.85rem",
                                fontFamily: "Poppins",
                              }}
                            >
                              {property.noofbalkanies} Balcony
                            </Typography>
                          </Stack>
                        )}
                      </Box>

                      {/* Price Section */}
                      <Box
                        sx={{
                          mt: "auto",
                          pt: 2,
                          borderTop: "1px solid #f0f0f0",
                        }}
                      >
                        <Stack
                          direction="row"
                          justifyContent="space-between"
                          alignItems="center"
                          mb={1.5}
                        >
                          <Box>
                            <Typography
                              variant="body2"
                              sx={{
                                color: "#666",
                                fontSize: "0.85rem",
                                fontFamily: "Poppins",
                              }}
                            >
                              Total Price
                            </Typography>
                            <Typography
                              variant="h5"
                              sx={{
                                fontWeight: 700,
                                color: "#1976d2",
                                fontSize: { xs: "1.3rem", md: "1.5rem" },
                                fontFamily: "Poppins",
                              }}
                            >
                              ₹ {formatPrice(property?.price)}
                            </Typography>
                          </Box>

                          <Stack
                            direction="row"
                            alignItems="center"
                            spacing={1}
                          >
                            <Avatar
                              sx={{
                                width: 32,
                                height: 32,
                                bgcolor: "#1976d2",
                                fontSize: "0.8rem",
                              }}
                            >
                              {item.seller?.sellerName?.charAt(0) || "O"}
                            </Avatar>
                            <Box>
                              <Typography
                                variant="body2"
                                sx={{
                                  fontWeight: 600,
                                  color: "#333",
                                  fontSize: "0.85rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                {item.seller?.sellerName || "Owner"}
                              </Typography>
                              <Typography
                                variant="caption"
                                sx={{
                                  color: "#999",
                                  fontSize: "0.7rem",
                                  fontFamily: "Poppins",
                                }}
                              >
                                Verified Seller
                              </Typography>
                            </Box>
                          </Stack>
                        </Stack>

                        {/* View Details Button */}
                        <Box sx={{ display: "flex", justifyContent: "center" }}>
                          <button
                            onClick={(e) => {
                              e.stopPropagation();
                              handleViewDetails(item.fkPostpropertyId);
                            }}
                            style={{
                              backgroundColor: "#1976d2",
                              color: "white",
                              border: "none",
                              padding: "10px 18px",
                              borderRadius: "8px",
                              fontSize: "14px",
                              fontWeight: 600,
                              cursor: "pointer",
                              fontFamily: "Poppins",
                              transition: "all 0.3s ease",
                              width: "100%",
                            }}
                          >
                            View Details
                          </button>
                        </Box>
                      </Box>
                    </CardContent>
                  </Card>
                </Grid>
              );
            })}
          </Grid>
        )}
      </Container>
    </div>
  );
};

export default ViewPropertyPage;
