import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  IconButton,
  TextField,
  Chip,
  Tooltip,
  TablePagination,
  InputAdornment,
} from "@mui/material";
import { Spin } from "antd";
import { Edit, CloudUpload, Search } from "@mui/icons-material";
import axios from "axios";
import { useNavigate } from "react-router-dom";
import { URL } from "../../config";
import Loading from "../../Components/Loading/Loading";

const ManageListings = () => {
  const [data, setData] = useState([]);
  const [filteredData, setFilteredData] = useState([]);
  const [searchText, setSearchText] = useState("");
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();
  const sellerId = localStorage.getItem("sellerId");

  const fetchData = async () => {
    try {
      const res = await axios.get(`${URL}/post-property-masters`);
      const filteredBySeller = res.data.filter(
        (item) => String(item.fksellerId) === String(sellerId)
      );
      setData(filteredBySeller.slice().reverse());
      setFilteredData(filteredBySeller.slice());
    } catch (error) {
      console.error(error);
      alert("Failed to fetch data");
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  const getPublishStatus = (record) => {
    const val =
      record.publish_Flag ??
      record.publish_Fag ??
      record.publishFlag ??
      record.Publish_Flag ??
      record.isPublished ??
      record.published ??
      null;

    return val !== null ? Number(val) : null;
  };

  const handleSearch = (value) => {
    setSearchText(value);
    const filtered = data.filter(
      (item) =>
        item.Property_Title?.toLowerCase().includes(value.toLowerCase()) ||
        String(item.price)?.includes(value) ||
        String(item.builduparea)?.includes(value)
    );
    setFilteredData([...filtered].reverse());
    setPage(0);
  };

  useEffect(() => {
    const timer = setTimeout(() => {
      setLoading(false);
    }, 1000);
    return () => clearTimeout(timer);
  }, []);

  if (loading) return <Loading fullscreen />;

  return (
    <>
      <Box sx={{ px: { xs: 1.5, sm: 1 }, pt: 1.3, minHeight: "10vh" }}>
        <Paper
          elevation={4}
          sx={{
            p: { xs: 2, sm: 3 },
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 50,
          }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              flexWrap: "wrap",
              gap: 2,
            }}
          >
            <Typography
              variant="h6"
              fontWeight={700}
              sx={{
                color: "#1f2937",
                fontSize: { xs: "1.1rem", sm: "1.45rem" },
                fontFamily: "Poppins",
              }}
            >
              Publish Property Listings
            </Typography>

            <TextField
              placeholder="Search properties..."
              variant="outlined"
              value={searchText}
              onChange={(e) => handleSearch(e.target.value)}
              size="small"
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search sx={{ color: "gray", fontFamily: "Poppins" }} />
                  </InputAdornment>
                ),
              }}
              sx={{
                bgcolor: "#fff",
                borderRadius: 2,
                boxShadow: 1,
                width: { xs: "100%", sm: 300 },
                fontFamily: "Poppins",
              }}
            />
          </Box>
        </Paper>
      </Box>

      <Box
        sx={{
          px: { xs: 1.5, sm: 1 },
          pt: 1.3,
          // bgcolor: "#f0f2f5",
          minHeight: "90vh",
          // marginTop: 2,
        }}
      >
        <Paper
          elevation={4}
          sx={{
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: "auto",
            overflowX: "auto", // horizontal scroll for small devices
          }}
        >
          <TableContainer
            component={Paper}
            sx={{
              borderRadius: 3,
              overflow: "auto",
              boxShadow: "0px 4px 20px rgba(0,0,0,0.04)",
            }}
          >
            <Table stickyHeader size="small">
              <TableHead>
                <TableRow sx={{ bgcolor: "#f3f4f6" }}>
                  {["ID","Title", "Price", "Area (sq.ft)", "Status", "Actions"].map(
                    (col) => (
                      <TableCell
                        key={col}
                        sx={{
                          fontWeight: 600,
                          fontSize: { xs: 14, sm: 18 },
                          padding: 2,
                          fontFamily: "Poppins",
                        }}
                      >
                        {col}
                      </TableCell>
                    )
                  )}
                </TableRow>
              </TableHead>
              <TableBody>
                {filteredData
                  .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                  .map((row,index) => {
                    const status = getPublishStatus(row);
                    const disableButtons = status === 1 || status === 2;

                    return (
                      <TableRow
                        key={row.postpropertyId}
                        hover
                        sx={{
                          transition: "all 0.2s ease",
                          "&:hover": {
                            bgcolor: "#f9fafb",
                            transform: { sm: "scale(1.002)" }, // disable scale on extra small screens
                          },
                        }}
                      >
                        <TableCell sx={{ fontSize: { xs: 12, sm: 14,fontFamily: "Poppins", } }}>
                                                {" "}
                                                {page * rowsPerPage + index + 1}
                                              </TableCell>
                        <TableCell>
                          <Typography
                            fontWeight={600}
                            fontSize={{ xs: 12, sm: 14, fontFamily: "Poppins" }}
                          >
                            {row.Property_Title}
                          </Typography>
                        </TableCell>
                        <TableCell>
                          <Chip
                            label={`₹${row.price}`}
                            sx={{
                              bgcolor: "#e8f5e9",
                              color: "#2e7d32",
                              fontWeight: 500,
                              fontSize: { xs: 10, sm: 12 },
                              fontFamily: "Poppins",
                            }}
                            size="small"
                          />
                        </TableCell>
                        <TableCell
                          sx={{
                            fontSize: { xs: 12, sm: 14, fontFamily: "Poppins" },
                          }}
                        >
                          {row.supperbuilduparea} sq.ft
                        </TableCell>
                        <TableCell>
                          {status === 1 ? (
                            <Chip
                              label="Pending"
                              sx={{
                                bgcolor: "#e3f2fd",
                                color: "#1976d2",
                                fontSize: { xs: 10, sm: 12 },
                                fontFamily: "Poppins",
                              }}
                              size="small"
                            />
                          ) : status === 2 ? (
                            <Chip
                              label="Published"
                              sx={{
                                bgcolor: "#e8f5e9",
                                color: "#2e7d32",
                                fontSize: { xs: 10, sm: 12 },
                                fontFamily: "Poppins",
                              }}
                              size="small"
                            />
                          ) : (
                            <Chip
                              label="Draft"
                              sx={{
                                bgcolor: "#fff3e0",
                                color: "#ef6c00",
                                fontSize: { xs: 10, sm: 12 },
                                fontFamily: "Poppins",
                              }}
                              size="small"
                            />
                          )}
                        </TableCell>
                        <TableCell>
                          <Tooltip
                            title={
                              <span style={{ fontFamily: "Poppins" }}>
                                {disableButtons
                                  ? "Action disabled"
                                  : "Edit Property"}
                              </span>
                            }
                          >
                            <span>
                              <IconButton
                                onClick={() =>
                                  navigate(
                                    `/dashboard/edit-property/${row.postpropertyId}`
                                  )
                                }
                                disabled={disableButtons}
                                sx={{
                                  color: disableButtons
                                    ? "grey.400"
                                    : "primary.main",
                                  bgcolor: disableButtons ? "" : "primary.50",
                                  "&:hover": {
                                    bgcolor: disableButtons
                                      ? ""
                                      : "primary.100",
                                  },
                                }}
                              >
                                <Edit fontSize="small" />
                              </IconButton>
                            </span>
                          </Tooltip>

                          <Tooltip
                            title={
                              disableButtons
                                ? "Action disabled"
                                : "Publish Property"
                            }
                          >
                            <span>
                              <IconButton
                                onClick={() =>
                                  navigate(
                                    `/dashboard/publish-property/${row.postpropertyId}`
                                  )
                                }
                                disabled={disableButtons}
                                sx={{
                                  color: disableButtons
                                    ? "grey.400"
                                    : "secondary.main",
                                  bgcolor: disableButtons ? "" : "secondary.50",
                                  "&:hover": {
                                    bgcolor: disableButtons
                                      ? ""
                                      : "secondary.100",
                                  },
                                }}
                              >
                                <CloudUpload fontSize="small" />
                              </IconButton>
                            </span>
                          </Tooltip>
                        </TableCell>
                      </TableRow>
                    );
                  })}

                {filteredData.length === 0 && (
                  <TableRow>
                    <TableCell
                      colSpan={5}
                      align="center"
                      sx={{ fontSize: { xs: 12, sm: 14 } }}
                    >
                      No properties found.
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
          </TableContainer>

          <TablePagination
            rowsPerPageOptions={[5, 7, 10]}
            component="div"
            count={filteredData.length}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={(e, newPage) => setPage(newPage)}
            onRowsPerPageChange={(e) => {
              setRowsPerPage(parseInt(e.target.value, 10));
              setPage(0);
            }}
            sx={{
              mt: 2,
              ".MuiTablePagination-toolbar": {
                justifyContent: "flex-end",
                flexWrap: "wrap",
              },
              ".MuiTablePagination-selectLabel, .MuiTablePagination-displayedRows":
                {
                  fontSize: { xs: 12, sm: 14 },
                },
            }}
          />
        </Paper>
      </Box>
    </>
  );
};

export default ManageListings;
