import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  IconButton,
  Tooltip,
  TablePagination,
  Chip,
  InputAdornment,
  TextField,
  Button,
  Dialog,
  DialogContent,
} from "@mui/material";
import { Spin } from "antd";
import { Edit, Delete, Search, Add } from "@mui/icons-material";
import axios from "axios";
import { URL } from "../../config";
import { useNavigate } from "react-router-dom";
import dayjs from "dayjs";
import { Close } from "@mui/icons-material";
import Loading from "../../Components/Loading/Loading";

const AdsManagement = () => {
  const [ads, setAds] = useState([]);
  const [filteredAds, setFilteredAds] = useState([]);
  const [searchText, setSearchText] = useState("");
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [loading, setLoading] = useState(true);
  const [previewOpen, setPreviewOpen] = useState(false);
  const [previewImg, setPreviewImg] = useState("");

  const navigate = useNavigate();
  const loggedInSellerId = localStorage.getItem("sellerId");

  const fetchAds = async () => {
    try {
      setLoading(true);
      const res = await axios.get(`${URL}/ads-managements`);
      const allAds = res.data.slice().reverse();

      const activeAds = allAds.filter((ad) => ad.isActive === 1);

      const sellerAds = activeAds.filter(
        (ad) => String(ad.fksellerId) === String(loggedInSellerId)
      );

      setAds(sellerAds);
      setFilteredAds(sellerAds);
    } catch (error) {
      console.error(error);
      alert("Failed to fetch ads");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAds();
  }, []);

  const handleSearch = (value) => {
    setSearchText(value);
    const filtered = ads.filter(
      (item) =>
        item.adsTitle?.toLowerCase().includes(value.toLowerCase()) ||
        String(item.adsStartDate)?.includes(value) ||
        String(item.adsEndDate)?.includes(value) ||
        String(item.adsManagementId)?.includes(value) ||
        item.adsTypeName?.toLowerCase().includes(value.toLowerCase())
    );
    setFilteredAds(filtered.slice().reverse());
    setPage(0);
  };

  const handleDelete = async (id) => {
    if (!window.confirm("Are you sure you want to delete this ad?")) return;
    try {
      await axios.patch(`${URL}/ads-managements/soft-delete/${id}`);
      fetchAds();
      alert("Ad deactivated successfully");
    } catch (error) {
      console.error(error);
      alert("Failed to deactivate ad");
    }
  };

  const handleAddAd = () => {
    navigate("/dashboard/ads-form");
  };

  // NEW FUNCTION FOR EDIT
  const handleEdit = (adsId) => {
    navigate(`/dashboard/ads-formupdate/${adsId}`);
  };

  useEffect(() => {
    const timer = setTimeout(() => {
      setLoading(false);
    }, 1000);
    return () => clearTimeout(timer);
  }, []);

  if (loading) {
    return <Loading fullscreen size={70} />;
  }

  return (
    <>
      {/* Header with Search and Add Button */}
      <Box sx={{ px: { xs: 1.5, sm: 1 }, pt: 1.3, minHeight: "10vh" }}>
        <Paper
          elevation={4}
          sx={{
            p: { xs: 2, sm: 3 },
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 50,
          }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              flexWrap: "wrap",
              gap: 2,
            }}
          >
            <Typography
              variant="h6"
              fontWeight={700}
              sx={{
                color: "#1f2937",
                fontSize: {
                  xs: "1.1rem",
                  sm: "1.45rem",
                  fontFamily: "Poppins",
                },
              }}
            >
              Ads Management
            </Typography>

            <Box sx={{ display: "flex", gap: 2, flexWrap: "wrap" }}>
              <TextField
                placeholder="Search ads..."
                variant="outlined"
                value={searchText}
                onChange={(e) => handleSearch(e.target.value)}
                size="small"
                InputProps={{
                  startAdornment: (
                    <InputAdornment position="start">
                      <Search sx={{ color: "gray", }} />
                    </InputAdornment>
                  ),
                }}
                sx={{
                  bgcolor: "#fff",
                  borderRadius: 2,
                  boxShadow: 1,
                  width: { xs: "100%", sm: 250 },
                  fontFamily: "Poppins",
                }}
              />

              {/* New Ad Button */}
              <Button
                variant="contained"
                color="primary"
                startIcon={<Add />}
                onClick={handleAddAd}
                sx={{
                  borderRadius: 2,
                  fontWeight: 600,
                  textTransform: "none",
                  fontFamily: "Poppins",
                  boxShadow: "0px 4px 12px rgba(0,0,0,0.1)",
                  "&:hover": {
                    boxShadow: "0px 6px 16px rgba(0,0,0,0.15)",
                  },
                }}
              >
                New Ads
              </Button>
            </Box>
          </Box>
        </Paper>
      </Box>

      {/* Table Section */}
      <Box
        sx={{
          px: { xs: 1.5, sm: 1 },
          pt: 1.3,
          // bgcolor: "#f0f2f5",
          minHeight: "90vh",
          // marginTop: 2,
        }}
      >
        <Paper
          elevation={4}
          sx={{
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: "auto",
            overflowX: "auto",
          }}
        >
          <TableContainer
            component={Paper}
            sx={{
              borderRadius: 3,
              overflow: "auto",
              boxShadow: "0px 4px 20px rgba(0,0,0,0.04)",
            }}
          >
            <Table stickyHeader size="small">
              <TableHead>
                <TableRow sx={{ bgcolor: "#f3f4f6" }}>
                  {[
                    "ID",
                    "Title",
                    "Ads Type",
                    "Start Date",
                    "End Date",
                    "Photo",
                    "Actions",
                  ].map((col) => (
                    <TableCell
                      key={col}
                      sx={{
                        fontWeight: 600,
                        fontSize: { xs: 14, sm: 18 },
                        padding: 2,
                        fontFamily: "Poppins",
                      }}
                    >
                      {col}
                    </TableCell>
                  ))}
                </TableRow>
              </TableHead>
              <TableBody>
                {filteredAds
                  .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                  .map((row, index) => (
                    <TableRow
                      key={row.adsManagementId}
                      hover
                      sx={{
                        transition: "all 0.2s ease",
                        "&:hover": {
                          bgcolor: "#f9fafb",
                          transform: { sm: "scale(1.002)" },
                        },
                      }}
                    >
                      <TableCell sx={{ fontSize: { xs: 12, sm: 14, fontFamily: "Poppins", } }}>
                        {" "}
                        {page * rowsPerPage + index + 1}
                      </TableCell>
                      <TableCell>
                        <Typography
                          fontWeight={600}
                          fontSize={{ xs: 12, sm: 14,fontFamily: "Poppins", }}
                        >
                          {row.adsTitle}
                        </Typography>
                      </TableCell>
                      <TableCell sx={{ fontSize: { xs: 12, sm: 14, fontFamily: "Poppins", } }}>
                        {row.adsType.adstypeName || "—"}
                      </TableCell>
                      <TableCell sx={{ fontSize: { xs: 12, sm: 14, fontFamily: "Poppins", } }}>
                        {" "}
                        {row.adsStartDate
                          ? dayjs(row.adsStartDate).format(
                              "DD-MMM-YYYY hh:mm A"
                            )
                          : "—"}
                      </TableCell>
                      <TableCell sx={{ fontSize: { xs: 12, sm: 14, fontFamily: "Poppins", } }}>
                        {" "}
                        {row.adsEndDate
                          ? dayjs(row.adsEndDate).format("DD-MMM-YYYY hh:mm A")
                          : "—"}
                      </TableCell>
                      <TableCell>
                        {row.adsPhoto ? (
                          <img
                            src={`${row.adsPhoto}`}
                            alt={row.adsTitle}
                            style={{
                              width: 50,
                              height: 50,
                              objectFit: "cover",
                              borderRadius: 6,
                              
                            }}
                            onClick={() => {
                              setPreviewImg(row.adsPhoto);
                              setPreviewOpen(true);
                            }}
                          />
                        ) : (
                          <Chip
                            label="No Photo"
                            sx={{
                              bgcolor: "#f5f5f5",
                              color: "gray",
                              fontSize: { xs: 10, sm: 12 },
                            }}
                            size="small"
                          />
                        )}
                      </TableCell>
                      <TableCell>
                        {/* Edit Button */}
                        <Tooltip title="Edit">
                          <IconButton
                            color="primary"
                            onClick={() => handleEdit(row.adsId)}
                            disabled={row.publish_Fag === 2}
                          >
                            <Edit fontSize="small" />
                          </IconButton>
                        </Tooltip>

                        {/* Delete Button */}
                        {/* <Tooltip title="Delete">
                          <IconButton color="error" onClick={() => handleDelete(row.adsId)}  disabled={row.publish_Fag === 2}>
                            <Delete fontSize="small" />
                          </IconButton>
                        </Tooltip> */}
                      </TableCell>
                    </TableRow>
                  ))}

                {filteredAds.length === 0 && (
                  <TableRow>
                    <TableCell
                      colSpan={8}
                      align="center"
                      sx={{ fontSize: { xs: 12, sm: 14 } }}
                    >
                      No ads found.
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
          </TableContainer>

          <TablePagination
            rowsPerPageOptions={[5, 7, 10]}
            component="div"
            count={filteredAds.length}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={(e, newPage) => setPage(newPage)}
            onRowsPerPageChange={(e) => {
              setRowsPerPage(parseInt(e.target.value, 10));
              setPage(0);
            }}
            sx={{
              mt: 2,
              ".MuiTablePagination-toolbar": {
                justifyContent: "flex-end",
                flexWrap: "wrap",
              },
              ".MuiTablePagination-selectLabel, .MuiTablePagination-displayedRows":
                {
                  fontSize: { xs: 12, sm: 14 },
                },
            }}
          />
        </Paper>
      </Box>
      {/* Preview Dialog */}
      <Dialog
        open={previewOpen}
        onClose={() => setPreviewOpen(false)}
        maxWidth="md"
      >
        {/* Close Button */}
        <IconButton
          onClick={() => setPreviewOpen(false)}
          sx={{
            position: "absolute",
            top: 8,
            right: 8,
            color: "white",
            backgroundColor: "rgba(0,0,0,0.5)",
            "&:hover": { backgroundColor: "rgba(0,0,0,0.7)" },
            zIndex: 2,
          }}
        >
          <Close />
        </IconButton>

        <DialogContent sx={{ p: 0 }}>
          <img
            src={previewImg}
            alt="Preview"
            style={{ width: "800px", height: "auto", borderRadius: 8 }}
          />
        </DialogContent>
      </Dialog>
    </>
  );
};

export default AdsManagement;
