import React, { useState, useEffect } from "react";
import { Card, Typography, Button, Tag, Row, Col, Space, Carousel, Modal } from "antd";
import {
  EnvironmentOutlined,
  CheckCircleFilled,
  PhoneOutlined,
  ExclamationCircleOutlined,
  CrownOutlined,
} from "@ant-design/icons";
import { useNavigate } from "react-router-dom";
import { Divider } from "@mui/material";
import axios from "axios";
import { URL } from "../../config";
import { message as antMessage } from "antd";
import PhoneModal from "../../Components/Modal/PhoneModal";
import OTPModal from "../../Components/Modal/OtpVerificationmodal";
import MembershipPromptModal from "../../Components/Modal/MembershipPromptModal/MembershipPromptModal"; // ✅ नवीन इम्पोर्ट

const { Title, Text, Paragraph } = Typography;

const PropertyCard = ({ property }) => {
  const navigate = useNavigate();
  const [user, setUser] = useState(null);
  const [activePackage, setActivePackage] = useState(null);
  const [viewCount, setViewCount] = useState(0);
  const [isCheckingLimit, setIsCheckingLimit] = useState(false);

  // ✅ PhoneModal States
  const [phoneModalOpen, setPhoneModalOpen] = useState(false);
  const [regData, setRegData] = useState({ contactno: "" });
  const [pendingPropertyId, setPendingPropertyId] = useState(null);

  // ✅ OTPModal States
  const [otpModalOpen, setOtpModalOpen] = useState(false);
  const [phoneNumber, setPhoneNumber] = useState("");

  // ✅ नवीन MembershipPromptModal स्टेट
  const [membershipModalOpen, setMembershipModalOpen] = useState(false);

  // Get user from localStorage
  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);
      
      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();
        
        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }
  }, []);

  // Fetch current view count
  useEffect(() => {
    if (user) {
      fetchViewCount();
    }
  }, [user]);

  // Function to fetch user's current view count
  const fetchViewCount = async () => {
    try {
      let url = "";
      
      if (user?.fkusertypeId === 1 && user?.buyerId) {
        url = `${URL}/log-properties/buyer/${user.buyerId}`;
      } else if (user?.fkusertypeId !== 1 && user?.sellerId) {
        url = `${URL}/log-properties/seller/${user.sellerId}`;
      }

      if (url) {
        const response = await axios.get(url);
        setViewCount(response.data.length);
      }
    } catch (error) {
      console.error("Error fetching view count:", error);
    }
  };

  // Function to check if user can view more properties
  const canViewProperty = () => {
    // If no package, allow viewing (or you can restrict this based on your business logic)
    if (!activePackage) {
      return true; // You can change this to false if you want to restrict views without package
    }

    const viewLimit = activePackage.propertyVisit || 0;
    
    // If unlimited views (0 or negative limit), allow
    if (viewLimit <= 0) {
      return true;
    }

    // Check if user has exceeded limit
    return viewCount < viewLimit;
  };

  // Function to show package upgrade modal
  const showUpgradeModal = () => {
    const viewLimit = activePackage?.propertyVisit || 0;
    
    Modal.confirm({
      title: (
        <div style={{ display: "flex", alignItems: "center", gap: 8 }}>
          <ExclamationCircleOutlined style={{ color: "#ff4d4f", fontSize: 24 }} />
          <span>View Limit Reached</span>
        </div>
      ),
      icon: null,
      content: (
        <div style={{ padding: "16px 0" }}>
          {activePackage ? (
            <>
              <p style={{ fontSize: 16, marginBottom: 12 }}>
                You've reached your property view limit of <strong>{viewLimit} views</strong> 
                for your <strong>{activePackage.packageName}</strong> package.
              </p>
              <p style={{ fontSize: 14, color: "#666", marginBottom: 8 }}>
                To view more properties, please upgrade your package or wait for your package to renew.
              </p>
            </>
          ) : (
            <>
              <p style={{ fontSize: 16, marginBottom: 12 }}>
                You don't have an active package.
              </p>
              <p style={{ fontSize: 14, color: "#666", marginBottom: 8 }}>
                Purchase a package to view properties and access premium features.
              </p>
            </>
          )}
          
          <div style={{ 
            marginTop: 16, 
            padding: 16, 
            background: "#f0f5ff",
            borderRadius: 8,
            border: "1px solid #d6e4ff"
          }}>
            <div style={{ display: "flex", alignItems: "center", gap: 8, marginBottom: 8 }}>
              <CrownOutlined style={{ color: "#1890ff", fontSize: 20 }} />
              <strong style={{ color: "#1890ff" }}>Upgrade to Lello Prime</strong>
            </div>
            <ul style={{ margin: 0, paddingLeft: 20, color: "#595959" }}>
              <li>Unlimited property views</li>
              <li>Save unlimited properties</li>
              <li>Direct contact with owners</li>
              <li>Priority customer support</li>
            </ul>
          </div>
        </div>
      ),
      okText: "Upgrade Package",
      cancelText: "Cancel",
      okButtonProps: {
        style: {
          backgroundColor: "#1890ff",
          borderColor: "#1890ff",
        }
      },
      width: 500,
      onOk: () => {
        navigate("/membership");
      },
    });
  };

  // Function to log property view
  const logPropertyView = async (propertyId) => {
    try {
      console.log("=== Starting Property View Log (PropertyCard) ===");
      console.log("Property ID:", propertyId);

      // Get user data from localStorage
      const storedUser = localStorage.getItem("user");
      const userTypeId = localStorage.getItem("userTypeId");
      const sellerId = localStorage.getItem("sellerId");
      const buyerId = localStorage.getItem("buyerId");
      const role = localStorage.getItem("role");

      console.log("Stored User:", storedUser);
      console.log("User Type ID:", userTypeId);
      console.log("Seller ID:", sellerId);
      console.log("Buyer ID:", buyerId);
      console.log("Role:", role);

      let finalUserId = null;
      let finalUserTypeId = null;
      let userObj = null;

      // Try to parse user object if it exists
      if (storedUser) {
        try {
          userObj = JSON.parse(storedUser);
          console.log("Parsed User Object:", userObj);
        } catch (e) {
          console.log("Error parsing user:", e);
        }
      }

      // ✅ येथे बदल केला आहे - आता आपण fkuserId वापरू
      // Determine user type and ID based on role
      if (role === "seller" || userTypeId === "3") {
        // Seller user type
        finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
        finalUserId = sellerId || (userObj && userObj.sellerId);
        console.log("User Type: Seller");
      } else if (role === "buyer" || userTypeId === "1") {
        // Buyer user type
        finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
        finalUserId = buyerId || (userObj && (userObj.buyerId || userObj.userId || userObj.id));
        console.log("User Type: Buyer");
      } else {
        // Try to determine from stored data
        if (sellerId) {
          finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
          finalUserId = sellerId;
        } else if (buyerId) {
          finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
          finalUserId = buyerId;
        } else if (userObj) {
          // Last attempt from user object
          if (userObj.sellerId) {
            finalUserTypeId = userObj?.fkuserId || 3; // ✅ बदल: fkuserId वापरा
            finalUserId = userObj.sellerId;
          } else if (userObj.buyerId) {
            finalUserTypeId = userObj?.fkuserId || 1; // ✅ बदल: fkuserId वापरा
            finalUserId = userObj.buyerId;
          }
        }
      }

      console.log("Final User ID:", finalUserId);
      console.log("Final User Type ID:", finalUserTypeId);

      // Check if we have required data
      if (!finalUserId || !finalUserTypeId) {
        console.log("❌ No user ID or user type found, skipping log");
        return;
      }

      // Get current date and time
      const now = new Date();
      const isoDateTime = now.toISOString();
      
      // Set time to start of day for the date field (keeping it as ISO format)
      const dateAtMidnight = new Date(now);
      dateAtMidnight.setHours(0, 0, 0, 0);
      const dateISO = dateAtMidnight.toISOString();
      
      // Format time as HH:MM:SS
      const time = now.toTimeString().split(" ")[0];

      console.log("Date (ISO):", dateISO);
      console.log("Time:", time);
      console.log("ISO DateTime:", isoDateTime);

      // Prepare log data according to LogProperty model
      const logData = {
        fkBuyerIdSellerId: parseInt(finalUserId),
        fkPostpropertyId: parseInt(propertyId),
        fkUsertypeId: parseInt(finalUserTypeId), // ✅ आता हे fkuserId वापरेल
        date: dateISO,
        time: time,
        isDefault: 0,
        isOn: 1,
        isActive: 1,
        createdDate: isoDateTime,
        createdBy: parseInt(finalUserId),
        lastChanged: isoDateTime,
      };

      console.log("Sending Log Data:", logData);
      console.log("API URL:", `${URL}/log-properties`);

      // Post to log-properties endpoint
      const response = await axios.post(`${URL}/log-properties`, logData);
      
      console.log("✅ Property view logged successfully");
      console.log("Response:", response.data);
      console.log("=== End Property View Log ===");
      
      // Update local view count
      setViewCount(prev => prev + 1);
      
    } catch (error) {
      console.error("❌ Error logging property view:");
      console.error("Error message:", error.message);
      console.error("Error response:", error.response?.data);
      console.error("Error status:", error.response?.status);
      console.log("=== End Property View Log (ERROR) ===");
      // Don't block navigation if logging fails
    }
  };

  // ✅ Handle PhoneModal close
  const handlePhoneModalClose = () => {
    setPhoneModalOpen(false);
    setPendingPropertyId(null);
    setRegData({ contactno: "" });
  };

  // ✅ Handle Phone submit - Close PhoneModal and Open OTPModal
  const handlePhoneSubmit = () => {
    setPhoneNumber(regData.contactno); // Store phone number for OTP modal
    setPhoneModalOpen(false); // Close phone modal
    setOtpModalOpen(true); // Open OTP modal
  };

  // ✅ Handle going back to PhoneModal from OTPModal
  const handleEditPhoneClick = () => {
    setOtpModalOpen(false); // Close OTP modal
    setPhoneModalOpen(true); // Reopen phone modal
  };

  // ✅ Handle OTP verification success
  const handleOtpSuccess = (userData) => {
    console.log("OTP Success - User Data:", userData);

    // Store user data
    localStorage.setItem("user", JSON.stringify(userData));
    setUser(userData);

    // Close OTP modal
    setOtpModalOpen(false);

    // Reload user from localStorage (in case it was updated)
    const storedUser = localStorage.getItem("user");
    if (storedUser) {
      const parsedUser = JSON.parse(storedUser);
      setUser(parsedUser);

      // Check if user has active package
      if (parsedUser.activePackage) {
        const packageEndDate = new Date(parsedUser.activePackage.endDate);
        const currentDate = new Date();

        if (packageEndDate > currentDate) {
          setActivePackage(parsedUser.activePackage);
        }
      }
    }

    // ✅ OTP व्हेरिफिकेशननंतर, जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
    if (!userData.activePackage) {
      setMembershipModalOpen(true);
      return;
    }

    // Navigate to the pending property if there is one
    if (pendingPropertyId) {
      navigate(`/Propertyinfo/${pendingPropertyId}`, {
        state: { id: pendingPropertyId },
      });
      setPendingPropertyId(null);
    }
  };

  // ✅ Handle OTPModal close
  const handleOtpModalClose = () => {
    setOtpModalOpen(false);
    setPhoneNumber("");
    setRegData({ contactno: "" });
    setPendingPropertyId(null);
  };

  // ✅ MembershipPromptModal हँडलर
  const handleMembershipModalClose = () => {
    setMembershipModalOpen(false);
  };

  // ✅ Updated handleClick to open PhoneModal for non-logged-in users
  const handleClick = async (e) => {
    e.preventDefault();
    console.log("Property Card clicked, Property ID:", property.id);
    
    // ✅ Check if user is logged in - if not, open PhoneModal
    if (!user) {
      setPendingPropertyId(property.id); // Store property ID
      setPhoneModalOpen(true); // Open phone modal
      return;
    }

    setIsCheckingLimit(true);

    try {
      // ✅ नवीन तर्क: जर युजरला अॅक्टिव्ह पॅकेज नाही, तर MembershipPromptModal दाखवा
      if (!activePackage) {
        setIsCheckingLimit(false);
        setMembershipModalOpen(true); // ✅ नवीन मोडल उघडा
        return;
      }

      // Check if user can view property
      if (!canViewProperty()) {
        setIsCheckingLimit(false);
        showUpgradeModal();
        return;
      }

      // Log the property view
      await logPropertyView(property.id);
      
      // Navigate to property details
      navigate(`/Propertyinfo/${property.id}`, { state: property });
    } catch (error) {
      console.error("Error handling property click:", error);
      antMessage.error("Something went wrong. Please try again.");
    } finally {
      setIsCheckingLimit(false);
    }
  };

  // Format price to display in Lac
  const formatPrice = (price) => {
    if (!price || price === "Price on request") return "Price on request";

    const amount = Number(price);

    if (amount >= 10000000) {
      return `₹ ${(amount / 10000000).toFixed(2)} Cr`;
    } else if (amount >= 100000) {
      return `₹ ${(amount / 100000).toFixed(2)} Lakh`;
    } else {
      return `₹ ${amount}`;
    }
  };

  // Format area
  const formatArea = (area) => {
    if (!area || area === "N/A") return "N/A";
    return area.replace("sqft", "q.ft");
  };

  return (
    <>
      <Card
        hoverable
        onClick={handleClick}
        style={{
          borderRadius: 12,
          boxShadow: "0 2px 12px rgba(0,0,0,0.08)",
          overflow: "hidden",
          border: "2px solid #f0f0f0",
          transition: "all 0.3s ease",
          opacity: isCheckingLimit ? 0.7 : 1,
          cursor: isCheckingLimit ? "wait" : "pointer",
        }}
        bodyStyle={{ padding: 0 }}
      >
        <Row gutter={0}>
          {/* ----------------------- LEFT IMAGE SLIDER ----------------------- */}
          <Col xs={24} sm={10} md={8} style={{ position: "relative" }}>
            <Carousel
              autoplay
              dots={true}
              arrows={false}
              autoplaySpeed={3000}
              infinite={true}
              pauseOnHover={true}
              style={{ width: "100%", height: 288 }}
            >
              {property?.photos && property.photos.length > 0 ? (
                property.photos.map((img, index) => (
                  <div key={index}>
                    <img
                      src={img}
                      alt={`${property.title} - Image ${index + 1}`}
                      style={{
                        width: "100%",
                        height: 315,
                        objectFit: "cover",
                      }}
                      onError={(e) => {
                        e.target.src = "/default.jpg";
                      }}
                    />
                  </div>
                ))
              ) : property?.photo ? (
                <div>
                  <img
                    src={property.photo}
                    style={{
                      width: "100%",
                      height: 315,
                      objectFit: "cover",
                    }}
                    onError={(e) => {
                      e.target.src = "/default.jpg";
                    }}
                  />
                </div>
              ) : (
                <div>
                  <img
                    src="/default.jpg"
                    style={{
                      width: "100%",
                      height: 315,
                      objectFit: "cover",
                    }}
                  />
                </div>
              )}
            </Carousel>

            {/* VERIFIED BADGE */}
            <Space
              style={{
                position: "absolute",
                top: 12,
                left: 12,
                gap: 8,
                zIndex: 10,
              }}
            >
              {property.tags?.includes("Verified") && (
                <Tag
                  style={{
                    backgroundColor: "#52c41a",
                    color: "white",
                    border: "none",
                    fontWeight: 600,
                    padding: "4px 12px",
                    borderRadius: 4,
                    display: "flex",
                    alignItems: "center",
                    gap: 4,
                  }}
                >
                  <CheckCircleFilled /> Verified
                </Tag>
              )}
            </Space>

            {/* View Limit Badge - Show remaining views if package exists */}
            {activePackage && activePackage.propertyVisit > 0 && (
              <Tag
                style={{
                  position: "absolute",
                  top: 12,
                  right: 12,
                  backgroundColor: viewCount >= activePackage.propertyVisit 
                    ? "#ff4d4f" 
                    : viewCount >= activePackage.propertyVisit * 0.8 
                    ? "#faad14" 
                    : "#1890ff",
                  color: "white",
                  border: "none",
                  fontWeight: 600,
                  padding: "4px 12px",
                  borderRadius: 4,
                  zIndex: 10,
                }}
              >
                {activePackage.propertyVisit - viewCount} views left
              </Tag>
            )}
          </Col>

          {/* ----------------------- RIGHT CONTENT ----------------------- */}
          <Col xs={24} sm={14} md={16}>
            <div>
              {/* TITLE */}
              <Title
                level={4}
                style={{
                  margin: 0,
                  marginBottom: 2,
                  fontSize: 20,
                  padding: "20px 0px 0px 20px",
                  fontFamily: "Poppins",
                }}
              >
                {property.title}
              </Title>

              {/* LOCATION */}
              <Text
                type="secondary"
                style={{
                  fontSize: 14,
                  display: "block",
                  marginBottom: 6,
                  padding: "0px 20px",
                  fontFamily: "Poppins",
                }}
              >
                <EnvironmentOutlined style={{ marginRight: 4 }} />
                {property.locality}, {property.location}
              </Text>

              {/* PRICE / AREA / STATUS */}
              <Row
                gutter={0}
                style={{
                  marginBottom: 4,
                  display: "flex",
                  alignItems: "center",
                  textAlign: "center",
                }}
              >
                {/* PRICE */}
                <Col xs={7}>
                  <Text
                    type="secondary"
                    style={{
                      fontSize: 12,
                      letterSpacing: 0.3,
                      textTransform: "uppercase",
                      fontFamily: "Poppins",
                    }}
                  >
                    Price
                  </Text>

                  <Title
                    level={4}
                    style={{
                      margin: "4px 0 0 0",
                      fontSize: 18,
                      fontWeight: 700,
                      fontFamily: "Poppins",
                    }}
                  >
                    {formatPrice(property.price)}
                  </Title>
                </Col>

                {/* DIVIDER */}
                <Divider
                  type="vertical"
                  style={{
                    width: "1px",
                    backgroundColor: "rgb(124 119 119)",
                    height: "70px",
                  }}
                />

                {/* AREA */}
                <Col xs={7}>
                  <Text
                    type="secondary"
                    style={{
                      fontSize: 12,
                      letterSpacing: 0.3,
                      textTransform: "uppercase",
                      fontFamily: "Poppins",
                    }}
                  >
                    Built-up Area
                  </Text>

                  <Title
                    level={4}
                    style={{
                      margin: "4px 0 0 0",
                      fontSize: 18,
                      fontWeight: 700,
                      fontFamily: "Poppins",
                    }}
                  >
                    {formatArea(property.area)}
                  </Title>
                </Col>

                {/* DIVIDER */}
                <Divider
                  type="vertical"
                  style={{
                    width: "1px",
                    backgroundColor: "rgb(124 119 119)",
                    height: "70px",
                  }}
                />

                {/* STATUS */}
                <Col xs={7}>
                  <Text
                    type="secondary"
                    style={{
                      fontSize: 12,
                      letterSpacing: 0.3,
                      textTransform: "uppercase",
                      fontFamily: "Poppins",
                    }}
                  >
                    Status
                  </Text>

                  <Title
                    level={4}
                    style={{
                      margin: "4px 0 0 0",
                      fontSize: 18,
                      fontWeight: 700,
                      fontFamily: "Poppins",
                    }}
                  >
                    Ready to Move
                  </Title>
                </Col>
              </Row>

              {/* DIVIDER */}
              <Divider
                type="horizontal"
                style={{
                  width: "1px",
                  backgroundColor: "rgb(124 119 119)",
                }}
              />

              {/* PROPERTY FEATURES IN ONE ROW */}
              <div
                style={{
                  margin: "4px 0",
                  padding: "0px 24px",
                  display: "flex",
                  alignItems: "center",
                  gap: 20,
                }}
              >
                {property.noofbedrooms && (
                  <div style={{ display: "flex", alignItems: "center", gap: 6 }}>
                    <div
                      style={{
                        padding: 6,
                        background: "#f5f5f5",
                        borderRadius: 6,
                        display: "flex",
                        justifyContent: "center",
                        alignItems: "center",
                        width: 32,
                        height: 32,
                      }}
                    >
                      🛏️
                    </div>
                    <Text
                      style={{
                        fontSize: 14,
                        color: "#333",
                        fontFamily: "Poppins",
                      }}
                    >
                      {property.noofbedrooms} Beds
                    </Text>
                  </div>
                )}

                {property.noofbathrooms && (
                  <div style={{ display: "flex", alignItems: "center", gap: 6 }}>
                    <div
                      style={{
                        padding: 6,
                        background: "#f5f5f5",
                        borderRadius: 6,
                        display: "flex",
                        justifyContent: "center",
                        alignItems: "center",
                        width: 32,
                        height: 32,
                      }}
                    >
                      🚿
                    </div>
                    <Text
                      style={{
                        fontSize: 14,
                        color: "#333",
                        fontFamily: "Poppins",
                      }}
                    >
                      {property.noofbathrooms} Baths
                    </Text>
                  </div>
                )}

                {property.noofbalkanies && (
                  <div style={{ display: "flex", alignItems: "center", gap: 6 }}>
                    <div
                      style={{
                        padding: 6,
                        background: "#f5f5f5",
                        borderRadius: 6,
                        display: "flex",
                        justifyContent: "center",
                        alignItems: "center",
                        width: 32,
                        height: 32,
                      }}
                    >
                      🏖️
                    </div>
                    <Text
                      style={{
                        fontSize: 14,
                        color: "#333",
                        fontFamily: "Poppins",
                      }}
                    >
                      {property.noofbalkanies} Balcony
                    </Text>
                  </div>
                )}
              </div>

              {/* DESCRIPTION */}
              <Paragraph
                style={{
                  marginBottom: 16,
                  fontSize: 14,
                  color: "#666",
                  lineHeight: 1.6,
                  padding: "0px 24px",
                  fontFamily: "Poppins",
                }}
                ellipsis={{ rows: 2 }}
              >
                {property.desc}
              </Paragraph>

              {/* FOOTER BAR */}
              <Row
                justify="end"
                align="middle"
                style={{
                  marginTop: 16,
                  paddingTop: 16,
                  borderTop: "1px solid #f0f0f0",
                  backgroundColor: "#3EBFBB21",
                  padding: "10px 24px",
                }}
              >
                <Space size={8}></Space>

                <Button
                  type="primary"
                  icon={<PhoneOutlined />}
                  style={{
                    backgroundColor: "#40C7C3",
                    borderColor: "#40C7C3",
                    borderRadius: 6,
                    fontWeight: 600,
                    height: 40,
                    paddingLeft: 20,
                    paddingRight: 20,
                  }}
                  onClick={(e) => {
                    e.stopPropagation();
                    // Handle contact action
                  }}
                >
                  Contact
                </Button>
              </Row>
            </div>
          </Col>
        </Row>
      </Card>

      {/* ✅ PhoneModal Component */}
      <PhoneModal
        open={phoneModalOpen}
        handleClose={handlePhoneModalClose}
        regData={regData}
        setRegData={setRegData}
        onSubmit={handlePhoneSubmit}
      />

      {/* ✅ OTPModal Component */}
      <OTPModal
        open={otpModalOpen}
        onClose={handleOtpModalClose}
        phone={phoneNumber}
        onEditPhoneClick={handleEditPhoneClick}
        handleOtpSuccess={handleOtpSuccess}
      />

      {/* ✅ MembershipPromptModal Component */}
      <MembershipPromptModal
        open={membershipModalOpen}
        onClose={handleMembershipModalClose}
      />
    </>
  );
};

export default PropertyCard;