import React, { useState, useEffect } from "react";
import {
  Form,
  Input,
  Select,
  Button,
  Upload,
  Radio,
  Steps,
  Progress,
  message,
  Row,
  Col,
} from "antd";
import {
  UploadOutlined,
  CheckCircleOutlined,
  HomeFilled,
} from "@ant-design/icons";
import axios from "axios";
import { toast } from "react-toastify";
import { useParams, useNavigate } from "react-router-dom";
import { Box, Paper, Typography } from "@mui/material";
import Ads from "../../assets/Ads/Ad-2.jpg";
import { URL } from "../../config";

const { Step } = Steps;

const PropetyFormNew = () => {
  const { postpropertyId } = useParams();
  const navigate = useNavigate();
  const [form] = Form.useForm();
  const [current, setCurrent] = useState(0);
  const [progress, setProgress] = useState(0);

  const [mainPropertyTypes, setMainPropertyTypes] = useState([]);
  const [propertyTypes, setPropertyTypes] = useState([]);
  const [PostpeopertyTypes, setPostpeopertyTypes] = useState([]);

  const [selectedmainPropertyTypes, setSelectedmainPropertyTypes] =
    useState(null);
  const [selectedPropertyType, setSelectedPropertyType] = useState(null);
  const [selectedPostpeopertyTypes, setSelectedpostpeopertyTypes] =
    useState(null);

  const [furnishingTypes, setFurnishingTypes] = useState([]);
  const [selectedFurnishing, setSelectedFurnishing] = useState(null);

  const [formData, setFormData] = useState({});
  const [fileList, setFileList] = useState([]);

  useEffect(() => {
    fetchDropdowns();
    if (postpropertyId) {
      fetchPropertyDetails(postpropertyId);
    }
  }, [postpropertyId]);

  const fetchDropdowns = async () => {
    try {
      const [mainCat, postTypes, propTypes, furnishing] = await Promise.all([
        axios.get(`${URL}/main-propertty-type-masters`),
        axios.get(`${URL}/post-type-masters`),
        axios.get(`${URL}/property-type-masters`),
        axios.get(`${URL}/furnishing-masters`),
      ]);
      setMainPropertyTypes(mainCat.data);
      setPostpeopertyTypes(postTypes.data);
      setPropertyTypes(propTypes.data);
      setFurnishingTypes(furnishing.data);
    } catch (err) {
      console.error("Error loading dropdowns", err);
    }
  };

  const fetchPropertyDetails = async (propertyId) => {
    try {
      const res = await axios.get(`${URL}/post-property-masters/${propertyId}`);
      const data = res.data;

      form.setFieldsValue({
        propertyTitle: data.Property_Title,
        price: data.price,
        areaSqft: data.builduparea,
        description: data.Property_Description,
        // Location fields
        city: data.fkcityId,
        locality: data.fklocalityId,
        subLocality: data.sublocality,
        apartmentSociety: data.apartmentNameAreaName,
        HouseNo: data.housenumber,
      });

      setSelectedmainPropertyTypes(data.fkmainpropertytypeId);
      setSelectedpostpeopertyTypes(data.fkpostpropertytypeId);
      setSelectedPropertyType(data.fkpropertytypeId);
      setSelectedFurnishing(data.fkfurnishingId);

      if (data.photo) {
        setFileList([
          {
            uid: "-1",
            name: "existing-image.jpg",
            status: "done",
            url: data.photo,
          },
        ]);
      }
    } catch (error) {
      console.error("Error fetching property:", error);
      toast.error("Failed to load property details!");
    }
  };

  const next = () => {
    if (
      current === 0 &&
      (!selectedmainPropertyTypes ||
        !selectedPostpeopertyTypes ||
        !selectedPropertyType)
    ) {
      message.error("Please fill all required fields in Step 1.");
      return;
    }
    form.validateFields().then((values) => {
      setFormData({ ...formData, ...values });
      const nextStep = current + 1;
      setCurrent(nextStep);
      setProgress(Math.round(((nextStep + 1) / steps.length) * 100));
    });
  };

  const prev = () => {
    const prevStep = current - 1;
    setCurrent(prevStep);
    setProgress(Math.round(((prevStep + 1) / steps.length) * 100));
  };

  const onFinish = async (values) => {
  const allvalues = { ...formData, ...values };

  const sellerId = localStorage.getItem("sellerId");
  if (!sellerId) {
    message.error("Seller ID not found. Please log in again.");
    return;
  }

  const formDataToSend = new FormData();
  
  // Basic Details
  formDataToSend.append("Property_Title", allvalues.propertyTitle || "");
  formDataToSend.append("Property_Description", allvalues.description || "");
  formDataToSend.append("price", Number(allvalues.price) || 0);
  
  // Property Types
  formDataToSend.append("fkpostpropertytypeId", Number(selectedPostpeopertyTypes));
  formDataToSend.append("fkmainpropertytypeId", Number(selectedmainPropertyTypes));
  formDataToSend.append("fkpropertytypeId", Number(selectedPropertyType));
  formDataToSend.append("fksellerId", Number(sellerId));
  
  // Property Profile - Rooms
  formDataToSend.append("noofbedrooms", Number(allvalues.Bedrooms) || 0);
  formDataToSend.append("noofbathrooms", Number(allvalues.Bathrooms) || 0);
  formDataToSend.append("noofbalkanies", Number(allvalues.Balconies) || 0);
  
  // Property Profile - Areas (Fixed: use correct field names)
  formDataToSend.append("supperbuilduparea", Number(allvalues.plotarea) || 0);
  formDataToSend.append("builduparea", Number(allvalues.Buildup) || 0);
  formDataToSend.append("carpetarea", Number(allvalues.carpetarea) || 0);
  
  // Furnishing (Fixed: append only once)
  formDataToSend.append("fkfurnishingtypeId", Number(selectedFurnishing) || 1);
  
  // Floor Details
  formDataToSend.append("totalfloor", Number(allvalues.totalfloor) || 0);
  
  // Location Details
  formDataToSend.append("fkcityId", allvalues.city || "");
  formDataToSend.append("fklocalityId", allvalues.locality || "");
  formDataToSend.append("sublocality", allvalues.subLocality || "");
  formDataToSend.append("apartmentNameAreaName", allvalues.apartmentSociety || "");
  formDataToSend.append("housenumber", Number(allvalues.HouseNo) || 0);

  // Required fields with default values
  formDataToSend.append("annualduespay", 0);
  formDataToSend.append("bookingamount", 0);
  formDataToSend.append("coverparking", 0);
  formDataToSend.append("openparking", 0);
  formDataToSend.append("expectedrental", 0);
  formDataToSend.append("maintenance", 0);
  formDataToSend.append("membershipcharge", 0);
  formDataToSend.append("video", "");
  
  // Default foreign keys (adjust based on your DB defaults)
  formDataToSend.append("fkavailablestatusId", 1);
  formDataToSend.append("fkfloorId", 1);
  formDataToSend.append("fkmaintnecemonthId", 1);
  formDataToSend.append("fkotherrooms", 1);
  formDataToSend.append("fkownershipId", 1);
  formDataToSend.append("fkuserId", 1);
  formDataToSend.append("publish_Fag", 1);
  formDataToSend.append("fkpublish_TypeId", 1);

  // Images
  fileList.forEach((file) => {
    if (file.originFileObj) {
      formDataToSend.append("photo", file.originFileObj);
    }
  });

  try {
    if (postpropertyId) {
      await axios.patch(
        `${URL}/post-property-masters/${postpropertyId}`,
        formDataToSend,
        { headers: { "Content-Type": "multipart/form-data" } }
      );
      toast.success("Property updated successfully!");
    } else {
      await axios.post(
        `${URL}/post-property-masters/uploads`,
        formDataToSend,
        {
          headers: { "Content-Type": "multipart/form-data" },
        }
      );
      toast.success("Property saved successfully!");
    }

    form.resetFields();
    setFileList([]);
    setSelectedmainPropertyTypes(null);
    setSelectedpostpeopertyTypes(null);
    setSelectedPropertyType(null);
    setSelectedFurnishing(null);
    setCurrent(0);
    setProgress(0);

    navigate("/dashboard/manage-listings");
  } catch (error) {
    console.error("Error saving property:", error);
    toast.error(error.response?.data?.error?.message || "Failed to save property!");
  }
};

  const steps = [
    {
      title: "Basic Details",
      content: (
        <>
          <Typography
            variant="h6"
            style={{ marginBottom: 16, fontWeight: 600 }}
          >
            Basic Details
          </Typography>

          <Form.Item
            name="propertyTitle"
            label="Property Title"
            rules={[{ required: true, message: "Please enter property title" }]}
          >
            <Input style={{ width: "70%" }} />
          </Form.Item>

          <Form.Item label="You're looking to ..." required>
            <div style={{ display: "flex", flexWrap: "wrap", gap: "10px" }}>
              {PostpeopertyTypes.map((option) => (
                <Button
                  key={option.id}
                  type={
                    selectedPostpeopertyTypes === option.postTypeId
                      ? "primary"
                      : "default"
                  }
                  onClick={() =>
                    setSelectedpostpeopertyTypes(option.postTypeId)
                  }
                >
                  {option.posttypeName}
                </Button>
              ))}
            </div>
          </Form.Item>

          <Form.Item label="And it's a ..." required>
            <Radio.Group
              value={selectedmainPropertyTypes}
              onChange={(e) => setSelectedmainPropertyTypes(e.target.value)}
            >
              {mainPropertyTypes.map((type) => (
                <Radio key={type.id} value={type.mainpropertytypeId}>
                  {type.mainprpoertyName}
                </Radio>
              ))}
            </Radio.Group>
          </Form.Item>

          <Form.Item label="Select property type" required>
            <div
              style={{
                display: "flex",
                flexWrap: "wrap",
                gap: "10px",
                width: "70%",
              }}
            >
              {propertyTypes.map((option) => (
                <Button
                  key={option.id}
                  type={
                    selectedPropertyType === option.propertyTypeId
                      ? "primary"
                      : "default"
                  }
                  onClick={() => setSelectedPropertyType(option.propertyTypeId)}
                >
                  {option.propertytypeName}
                </Button>
              ))}
            </div>
          </Form.Item>
        </>
      ),
    },
    {
      title: "Location Details",
      content: (
        <>
          <Typography
            variant="h6"
            style={{ marginBottom: 16, fontWeight: 600 }}
          >
            Location Details
          </Typography>

          <Form.Item
            name="city"
            label="City"
            rules={[{ required: true, message: "Please enter city" }]}
          >
            <Input placeholder="Enter city" style={{ width: "70%" }} />
          </Form.Item>
          <Form.Item
            name="locality"
            label="Locality"
            rules={[{ required: true, message: "Please enter locality" }]}
          >
            <Input placeholder="Enter locality" style={{ width: "70%" }} />
          </Form.Item>
          <Form.Item name="subLocality" label="Sub Locality (Optional)">
            <Input placeholder="Enter sub locality" style={{ width: "70%" }} />
          </Form.Item>
          <Form.Item
            name="apartmentSociety"
            label="Apartment / Society (Optional)"
          >
            <Input
              placeholder="Enter apartment or society name"
              style={{ width: "70%" }}
            />
          </Form.Item>
          <Form.Item name="HouseNo" label="House No (Optional)">
            <Input placeholder="Enter House No" style={{ width: "70%" }} />
          </Form.Item>
        </>
      ),
    },
    {
      title: "Property Profile",
      content: (
        <>
          <Typography
            variant="h6"
            style={{ marginBottom: 16, fontWeight: 600 }}
          >
            Property Profile
          </Typography>

          <Form.Item
            name="Bedrooms"
            label="No. of Bedrooms"
            rules={[{ required: true, message: "Please enter Bedrooms" }]}
          >
            <Input
              type="number"
              placeholder="Enter No. Of Bedrooms"
              style={{ width: "70%" }}
            />
          </Form.Item>

          <Form.Item
            name="Bathrooms"
            label="No. of Bathrooms"
            rules={[{ required: true, message: "Please enter Bathrooms" }]}
          >
            <Input
              type="number"
              placeholder="Enter No. of Bathrooms"
              style={{ width: "70%" }}
            />
          </Form.Item>

          <Form.Item name="Balconies" label="Balconies">
            <Input
              type="number"
              placeholder="Enter No. Balconies"
              style={{ width: "70%" }}
            />
          </Form.Item>

          <Form.Item
            name="plotarea"
            label="Plot Area (sq.ft)"
            rules={[
              {
                validator: (_, value) =>
                  !value || Number(value) > 0
                    ? Promise.resolve()
                    : Promise.reject(
                        new Error("Area must be a positive number")
                      ),
              },
            ]}
          >
            <Input type="number" style={{ width: "70%" }} />
          </Form.Item>

          <Form.Item
            name="Buildup"
            label="Build-up Area (sq.ft)"
            rules={[
              {
                validator: (_, value) =>
                  !value || Number(value) > 0
                    ? Promise.resolve()
                    : Promise.reject(
                        new Error("Area must be a positive number")
                      ),
              },
            ]}
          >
            <Input type="number" style={{ width: "70%" }} />
          </Form.Item>

          <Form.Item
            name="carpetarea"
            label="Carpet Area (sq.ft)"
            rules={[
              {
                validator: (_, value) =>
                  !value || Number(value) > 0
                    ? Promise.resolve()
                    : Promise.reject(
                        new Error("Area must be a positive number")
                      ),
              },
            ]}
          >
            <Input type="number" style={{ width: "70%" }} />
          </Form.Item>

          <Form.Item label="Furnishing" required>
            <div style={{ display: "flex", flexWrap: "wrap", gap: "10px" }}>
              {furnishingTypes.map((item) => (
                <Button
                  key={item.id}
                  type={
                    selectedFurnishing === item.furnishingId
                      ? "primary"
                      : "default"
                  }
                  onClick={() => setSelectedFurnishing(item.furnishingId)}
                >
                  {item.furnishingName}
                </Button>
              ))}
            </div>
          </Form.Item>

          <Form.Item name="totalfloor" label="Total Floor">
            <Input placeholder="Enter Total Floor" style={{ width: "70%" }} />
          </Form.Item>
        </>
      ),
    },

    {
      title: "Property Features",
      content: (
        <>
          <Typography
            variant="h6"
            style={{ marginBottom: 16, fontWeight: 600 }}
          >
            Property Features
          </Typography>

          <Form.Item
            name="price"
            label="Price"
            rules={[
              { required: true, message: "Please enter price" },
              {
                type: "number",
                min: 0,
                transform: (value) => Number(value),
                message: "Price must be a positive number",
              },
            ]}
          >
            <Input type="number" style={{ width: "70%" }} />
          </Form.Item>
        </>
      ),
    },
    {
      title: "Media & Submit",
      content: (
        <>
          <Typography
            variant="h6"
            style={{ marginBottom: 16, fontWeight: 600 }}
          >
            Media & Submit
          </Typography>

          <Form.Item
            name="images"
            label="Upload Images"
            style={{ width: "70%" }}
            rules={[
              {
                validator: (_, value) =>
                  fileList && fileList.length > 0
                    ? Promise.resolve()
                    : Promise.reject(
                        new Error("Please upload at least one image")
                      ),
              },
            ]}
          >
            <Upload
              listType="picture"
              beforeUpload={() => false} // prevent automatic upload
              fileList={fileList}
              onChange={({ fileList: newFileList }) => setFileList(newFileList)}
              multiple
              accept="image/*"
            >
              <Button icon={<UploadOutlined />}>Click to Upload</Button>
            </Upload>
          </Form.Item>

          <Form.Item name="description" label="Description">
            <Input.TextArea rows={4} style={{ width: "70%" }} />
          </Form.Item>
        </>
      ),
    },
  ];

  return (
    <>
      {/* Header */}
      <Box
        sx={{
          px: { xs: 1.5, sm: 3 },
          pt: 3,
          minHeight: "10vh",
        }}
      >
        <Paper
          elevation={4}
          sx={{
            p: { xs: 2, sm: 2 },
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 50,
          }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
            }}
          >
            {/* Left side title */}
            <Typography
              variant="h6"
              fontWeight={700}
              sx={{
                color: "#1f2937",
                fontSize: { xs: "1.1rem", sm: "1.45rem" },
                display: "flex",
                alignItems: "center",
                gap: 1,
              }}
            >
              Post Your Property
            </Typography>

            {/* Right side AntD Dashboard Progress */}
            <Progress
              type="dashboard"
              percent={Math.round((current / (steps.length - 1)) * 100)}
              width={80}
              size={50}
              strokeWidth={15}
              steps={4}
              trailColor="rgba(0, 0, 0, 0.06)"
            />
          </Box>
        </Paper>
      </Box>

      {/* Body */}
      <Box
        sx={{
          px: { xs: 1.5, sm: 3 },
          pt: 2,
        }}
      >
        <Paper
          elevation={4}
          sx={{
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 90,
            p: 1,
            px: 2,
          }}
        >
          {/* Stepper */}
          <Steps current={current} style={{ margin: "20px 0px" }}>
            {steps.map((item) => (
              <Step key={item.title} title={item.title} />
            ))}
          </Steps>
        </Paper>
      </Box>

      <Box
        sx={{
          px: { xs: 1.5, sm: 3 },
          pt: 2,
          minHeight: "90vh",
        }}
      >
        <Paper
          elevation={4}
          sx={{
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 550,
            p: 3,
          }}
        >
          {/* Form with Ads */}
          <Row justify="center" gutter={[24, 24]}>
            <Col xs={24} sm={24} md={14} lg={14}>
              <Form
                form={form}
                layout="vertical"
                onFinish={current === steps.length - 1 ? onFinish : undefined}
              >
                {steps[current].content}

                <Form.Item>
                  <Box
                    sx={{
                      display: "flex",
                      justifyContent: "space-between",
                      width: "70%",
                    }}
                  >
                    {current > 0 && (
                      <Button
                        type="primary"
                        onClick={prev}
                        style={{
                          padding: "8px 28px",
                          minWidth: "120px",
                          fontWeight: 600,
                          borderRadius: "2px",
                          boxShadow: "0 4px 8px rgba(0,0,0,0.15)",
                          transition: "all 0.3s ease",
                        }}
                        onMouseOver={(e) => {
                          e.currentTarget.style.transform = "translateY(-2px)";
                          e.currentTarget.style.boxShadow =
                            "0 6px 12px rgba(0,0,0,0.2)";
                        }}
                        onMouseOut={(e) => {
                          e.currentTarget.style.transform = "translateY(0)";
                          e.currentTarget.style.boxShadow =
                            "0 4px 8px rgba(0,0,0,0.15)";
                        }}
                      >
                        ← Previous
                      </Button>
                    )}
                    {current < steps.length - 1 && (
                      <Button
                        type="primary"
                        onClick={next}
                        disabled={
                          current === 0 &&
                          (!selectedmainPropertyTypes ||
                            !selectedPostpeopertyTypes ||
                            !selectedPropertyType)
                        }
                        style={{
                          padding: "8px 28px",
                          minWidth: "120px",
                          fontWeight: 600,
                          borderRadius: "2px",
                          boxShadow: "0 4px 8px rgba(0,0,0,0.15)",
                          transition: "all 0.3s ease",
                          marginLeft: "10px",
                        }}
                        onMouseOver={(e) => {
                          e.currentTarget.style.transform = "translateY(-2px)";
                          e.currentTarget.style.boxShadow =
                            "0 6px 12px rgba(0,0,0,0.2)";
                        }}
                        onMouseOut={(e) => {
                          e.currentTarget.style.transform = "translateY(0)";
                          e.currentTarget.style.boxShadow =
                            "0 4px 8px rgba(0,0,0,0.15)";
                        }}
                      >
                        Next →
                      </Button>
                    )}
                    {current === steps.length - 1 && (
                      <Button
                        type="primary"
                        htmlType="submit"
                        icon={<CheckCircleOutlined />}
                        style={{
                          padding: "10px 24px",
                          minWidth: "150px",
                          fontWeight: 600,
                          borderRadius: "2px",
                          background:
                            "linear-gradient(90deg, #28a745, #34d058)",
                          border: "none",
                          boxShadow: "0 4px 10px rgba(40, 167, 69, 0.3)",
                          display: "flex",
                          alignItems: "center",
                          justifyContent: "center",
                          gap: "8px",
                          transition: "all 0.3s ease",
                        }}
                        onMouseEnter={(e) => {
                          e.currentTarget.style.background =
                            "linear-gradient(90deg, #34d058, #50fa7b)";
                          e.currentTarget.style.boxShadow =
                            "0 6px 14px rgba(40, 167, 69, 0.4)";
                        }}
                        onMouseLeave={(e) => {
                          e.currentTarget.style.background =
                            "linear-gradient(90deg, #28a745, #34d058)";
                          e.currentTarget.style.boxShadow =
                            "0 4px 10px rgba(40, 167, 69, 0.3)";
                        }}
                      >
                        {postpropertyId ? "Update" : "Submit"}
                      </Button>
                    )}
                  </Box>
                </Form.Item>
              </Form>
            </Col>

            <Col
              xs={24}
              sm={24}
              md={10}
              lg={10}
              style={{
                background: "rgb(245, 245, 245)",
                display: "flex",
                justifyContent: "center",
                alignItems: "center",
                paddingBottom: "10px",
              }}
            >
              <img
                src={Ads}
                alt="Property Banner"
                style={{
                  maxWidth: "100%",
                  height: "600px",
                  borderRadius: "8px",
                }}
              />
            </Col>
          </Row>
        </Paper>
      </Box>
    </>
  );
};




export default PropetyFormNew