import React, { useState, useEffect } from "react";
import {
  Form,
  Input,
  Select,
  Button,
  Upload,
  Radio,
  Steps,
  Progress,
  message,
  Row,
  Col,
  Divider,
  AutoComplete,
} from "antd";
import {
  UploadOutlined,
  CheckCircleOutlined,
  HomeOutlined,
  EnvironmentOutlined,
  FileImageOutlined,
  DollarOutlined,
  InfoCircleOutlined,
  CameraOutlined,
  PlusOutlined,
  VideoCameraOutlined,
} from "@ant-design/icons";
import axios from "axios";
import { toast } from "react-toastify";
import { useParams, useNavigate } from "react-router-dom";
import { Box, Paper, Typography } from "@mui/material";
import Ads from "../../assets/Ads/Ad-2.jpg";
import { URL } from "../../config";
import usePlacesAutocomplete, {
  getGeocode,
  getLatLng,
} from "use-places-autocomplete";

const { Step } = Steps;
const { Option } = Select;

const PropertyForm3 = () => {
  const { postpropertyId } = useParams();
  const navigate = useNavigate();
  const [form] = Form.useForm();
  const [current, setCurrent] = useState(0);
  const [progress, setProgress] = useState(0);

  const [mainPropertyTypes, setMainPropertyTypes] = useState([]);
  const [propertyTypes, setPropertyTypes] = useState([]);
  const [PostpeopertyTypes, setPostpeopertyTypes] = useState([]);

  const [selectedmainPropertyTypes, setSelectedmainPropertyTypes] =
    useState(null);
  const [selectedPropertyType, setSelectedPropertyType] = useState(null);
  const [selectedPostpeopertyTypes, setSelectedpostpeopertyTypes] =
    useState(null);

  const [furnishingTypes, setFurnishingTypes] = useState([]);
  const [selectedFurnishing, setSelectedFurnishing] = useState(null);

  const [imagesFileList, setImagesFileList] = useState([]);
  const [videoFileList, setVideoFileList] = useState([]);
  const [existingVideo, setExistingVideo] = useState(null);

  const [formData, setFormData] = useState({});
  const [fileList, setFileList] = useState([]);

  useEffect(() => {
    fetchDropdowns();
    if (postpropertyId) {
      fetchPropertyDetails(postpropertyId);
    }
  }, [postpropertyId]);

  const fetchDropdowns = async () => {
    try {
      const [mainCat, postTypes, propTypes, furnishing] = await Promise.all([
        axios.get(`${URL}/main-propertty-type-masters`),
        axios.get(`${URL}/post-type-masters`),
        axios.get(`${URL}/property-type-masters`),
        axios.get(`${URL}/furnishing-masters`),
      ]);
      setMainPropertyTypes(mainCat.data);
      setPostpeopertyTypes(postTypes.data);
      setPropertyTypes(propTypes.data);
      setFurnishingTypes(furnishing.data);
    } catch (err) {
      console.error("Error loading dropdowns", err);
    }
  };

  const fetchPropertyDetails = async (propertyId) => {
    try {
      const res = await axios.get(`${URL}/post-property-masters/${propertyId}`);
      const data = res.data;

      form.setFieldsValue({
        propertyTitle: data.Property_Title,
        price: data.price,
        pricepersqft: data.pricepersqft,
        description: data.Property_Description,
        // Location fields
        city: data.fkcityId,
        locality: data.fklocalityId,
        subLocality: data.sublocality,
        apartmentSociety: data.apartmentNameAreaName,
        HouseNo: data.housenumber,
        Bedrooms: data.noofbedrooms,
        Bathrooms: data.noofbathrooms,
        Balconies: data.noofbalkanies,
        supperbuilduparea: data.supperbuilduparea,
        carpetarea: data.carpetarea,
        Buildup: data.builduparea,
        totalfloor: data.totalfloor,
      });

      setSelectedmainPropertyTypes(data.fkmainpropertytypeId);
      setSelectedpostpeopertyTypes(data.fkpostpropertytypeId);
      setSelectedPropertyType(data.fkpropertytypeId);
      setSelectedFurnishing(data.fkfurnishingtypeId);

      if (data.photo) {
        setFileList([
          {
            uid: "-1",
            name: "existing-image.jpg",
            status: "done",
            url: data.photo,
          },
        ]);
      }

      if (data.video) {
        setExistingVideo(data.video);
        setVideoFileList([
          {
            uid: "-2",
            name: "existing-video.mp4",
            status: "done",
            url: data.video,
          },
        ]);
      }
    } catch (error) {
      console.error("Error fetching property:", error);
      toast.error("Failed to load property details!");
    }
  };

  // Google Places Autocomplete
  const {
    ready,
    value,
    suggestions: { status, data },
    setValue,
    clearSuggestions,
  } = usePlacesAutocomplete({
    requestOptions: {
      types: ["(cities)"], // फक्त शहरांसाठी
      componentRestrictions: { country: "in" }, // भारतापुरतं
    },
    debounce: 300,
  });

  const handleSelectCity = async (cityName) => {
    setValue(cityName, false);
    clearSuggestions();
    form.setFieldsValue({ city: cityName });

    try {
      const results = await getGeocode({ address: cityName });
      const { lat, lng } = await getLatLng(results[0]);
      console.log("City Coordinates:", lat, lng);

      // शहराचे नाव आणि कोऑर्डिनेट्स स्टोअर करण्यासाठी
      form.setFieldsValue({
        city: cityName,
        cityLat: lat,
        cityLng: lng,
      });
    } catch (error) {
      console.error("Error fetching city coordinates:", error);
    }
  };

  const next = () => {
    if (
      current === 0 &&
      (!selectedmainPropertyTypes ||
        !selectedPostpeopertyTypes ||
        !selectedPropertyType)
    ) {
      message.error("Please fill all required fields in Step 1.");
      return;
    }
    form.validateFields().then((values) => {
      setFormData({ ...formData, ...values });
      const nextStep = current + 1;
      setCurrent(nextStep);
      setProgress(Math.round(((nextStep + 1) / steps.length) * 100));
    });
  };

  const prev = () => {
    const prevStep = current - 1;
    setCurrent(prevStep);
    setProgress(Math.round(((prevStep + 1) / steps.length) * 100));
  };

  const onFinish = async (values) => {
    const allvalues = { ...formData, ...values };

    const sellerId = localStorage.getItem("sellerId");
    if (!sellerId) {
      message.error("Seller ID not found. Please log in again.");
      return;
    }

    const formDataToSend = new FormData();
    formDataToSend.append("Property_Title", allvalues.propertyTitle);
    formDataToSend.append("Property_Description", allvalues.description || "");
    formDataToSend.append("price", Number(allvalues.price));
    formDataToSend.append("pricepersqft", Number(allvalues.pricepersqft || ""));
    formDataToSend.append("fkpostpropertytypeId", selectedPostpeopertyTypes);
    formDataToSend.append("fkmainpropertytypeId", selectedmainPropertyTypes);
    formDataToSend.append("fkpropertytypeId", selectedPropertyType);
    formDataToSend.append("fksellerId", Number(sellerId));

    // Location fields
    formDataToSend.append("fkcityId", allvalues.city || "");
    formDataToSend.append("fklocalityId", allvalues.locality || "");
    formDataToSend.append("sublocality", allvalues.subLocality || "");
    formDataToSend.append(
      "apartmentNameAreaName",
      allvalues.apartmentSociety || ""
    );
    formDataToSend.append("housenumber", allvalues.HouseNo || "");

    // ✅ PROPERTY PROFILE FIELDS
    formDataToSend.append("noofbedrooms", Number(allvalues.Bedrooms || 0));
    formDataToSend.append("noofbathrooms", Number(allvalues.Bathrooms || 0));
    formDataToSend.append("noofbalkanies", Number(allvalues.Balconies || 0));
    formDataToSend.append(
      "supperbuilduparea",
      Number(allvalues.supperbuilduparea || 0)
    );
    formDataToSend.append("carpetarea", Number(allvalues.carpetarea || 0));
    formDataToSend.append("builduparea", Number(allvalues.Buildup || 0));
    formDataToSend.append("totalfloor", Number(allvalues.totalfloor || 0));
    formDataToSend.append(
      "fkfurnishingtypeId",
      Number(selectedFurnishing || 0)
    );

    fileList.forEach((file) => {
      if (file.originFileObj) {
        formDataToSend.append("photo", file.originFileObj);
      }
    });

    // ✅ Append Video
    if (videoFileList.length > 0 && videoFileList[0].originFileObj) {
      formDataToSend.append("video", videoFileList[0].originFileObj);
    }

    try {
      if (postpropertyId) {
        await axios.patch(
          `${URL}/post-property-masters/${postpropertyId}`,
          formDataToSend,
          { headers: { "Content-Type": "multipart/form-data" } }
        );
        toast.success("Property updated successfully!");
      } else {
        await axios.post(
          `${URL}/post-property-masters/uploads`,
          formDataToSend,
          {
            headers: { "Content-Type": "multipart/form-data" },
          }
        );
        toast.success("Property saved successfully!");
      }

      form.resetFields();
      setFileList([]);
      setSelectedmainPropertyTypes(null);
      setSelectedpostpeopertyTypes(null);
      setSelectedPropertyType(null);
      setCurrent(0);
      setProgress(0);

      navigate("/dashboard/manage-listings");
    } catch (error) {
      toast.error("Failed to save property!");
    }
  };

  const steps = [
    {
      title: <span style={{ fontFamily: "Poppins" }}>Basic Details</span>,
      icon: <HomeOutlined />,
      content: (
        <>
          <Typography
            variant="h4"
            style={{ marginBottom: 4, fontWeight: 500, fontFamily: "Poppins" }}
          >
            <HomeOutlined /> Basic Details
          </Typography>
          <Typography
            type="secondary"
            style={{
              fontSize: "14px",
              marginBottom: 16,
              fontFamily: "Poppins",
              color: "#778899",
            }}
          >
            Provide the basic information about your property
          </Typography>

          <Form.Item
            name="propertyTitle"
            label={
              <span style={{ fontFamily: "Poppins" }}>Property Title</span>
            }
            rules={[{ required: true, message: "Please enter property title" }]}
          >
            <Input
              placeholder="Enter a descriptive title for your property"
              style={{ width: "100%", fontFamily: "Poppins" }}
            />
          </Form.Item>

          <Form.Item
            label={
              <span style={{ fontFamily: "Poppins" }}>
                You're looking to ...
              </span>
            }
            required
          >
            <div style={{ display: "flex", flexWrap: "wrap", gap: "10px" }}>
              {PostpeopertyTypes.map((option) => (
                <Button
                  key={option.id}
                  style={{ fontFamily: "Poppins" }}
                  type={
                    selectedPostpeopertyTypes === option.postTypeId
                      ? "primary"
                      : "default"
                  }
                  onClick={() =>
                    setSelectedpostpeopertyTypes(option.postTypeId)
                  }
                >
                  {option.posttypeName}
                </Button>
              ))}
            </div>
          </Form.Item>

          <Form.Item
            label={
              <span style={{ fontFamily: "Poppins" }}>And it's a ...</span>
            }
            required
          >
            <Radio.Group
              value={selectedmainPropertyTypes}
              onChange={(e) => setSelectedmainPropertyTypes(e.target.value)}
            >
              {mainPropertyTypes.map((type) => (
                <Radio
                  key={type.id}
                  style={{ fontFamily: "Poppins" }}
                  value={type.mainpropertytypeId}
                >
                  {type.mainprpoertyName}
                </Radio>
              ))}
            </Radio.Group>
          </Form.Item>

          <Form.Item
            label={
              <span style={{ fontFamily: "Poppins" }}>
                Select property type
              </span>
            }
            required
          >
            <div
              style={{
                display: "flex",
                flexWrap: "wrap",
                gap: "10px",
              }}
            >
              {propertyTypes.map((option) => (
                <Button
                  key={option.id}
                  type={
                    selectedPropertyType === option.propertyTypeId
                      ? "primary"
                      : "default"
                  }
                  onClick={() => setSelectedPropertyType(option.propertyTypeId)}
                  style={{ fontFamily: "Poppins" }}
                >
                  {option.propertytypeName}
                </Button>
              ))}
            </div>
          </Form.Item>
        </>
      ),
    },
    {
      title: <span style={{ fontFamily: "Poppins" }}>Location Details</span>,
      icon: <EnvironmentOutlined />,
      content: (
        <>
          <Typography
            variant="h4"
            style={{ marginBottom: 6, fontWeight: 500, fontFamily: "Poppins" }}
          >
            <EnvironmentOutlined /> Location Details
          </Typography>
          <Typography
            type="secondary"
            style={{
              fontSize: "14px",
              marginBottom: 16,
              fontFamily: "Poppins",
              color: "#778899",
            }}
          >
            Help buyers find your property with accurate location information
          </Typography>

          <Row gutter={[16, 16]}>
            <Col xs={24} sm={24} md={12}>
              {/* <Form.Item
                name="city"
               
                  label={<span style={{ fontFamily: "Poppins" }}>City</span>}
                rules={[{ required: true, message: "Please enter city" }]}
                style={{ fontFamily: "Poppins" }}
              >
                <Input
                  placeholder="Enter city"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item> */}
              <Form.Item
                name="city"
                label={<span style={{ fontFamily: "Poppins" }}>City</span>}
                rules={[{ required: true, message: "Please select city" }]}
                style={{ fontFamily: "Poppins" }}
              >
                <AutoComplete
                  style={{ width: "100%", fontFamily: "Poppins" }}
                  onSearch={(value) => setValue(value)}
                  onSelect={(value) => {
                    handleSelectCity(value);
                    form.setFieldsValue({ city: value });
                  }}
                  placeholder="Search city"
                  value={value}
                  options={
                    status === "OK"
                      ? data.map(({ place_id, description }) => ({
                          value: description,
                          label: description,
                          key: place_id,
                        }))
                      : []
                  }
                />
              </Form.Item>
            </Col>
            <Col xs={24} sm={24} md={12}>
              <Form.Item
                name="locality"
                label={<span style={{ fontFamily: "Poppins" }}>Locality</span>}
                rules={[{ required: true, message: "Please enter locality" }]}
                style={{ fontFamily: "Poppins" }}
              >
                <Input
                  placeholder="Enter locality"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item>
            </Col>
          </Row>

          <Row gutter={[16, 16]}>
            <Col xs={24} sm={24} md={12}>
              <Form.Item
                name="subLocality"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    Sub Locality (Optional)
                  </span>
                }
                style={{ fontFamily: "Poppins" }}
              >
                <Input
                  placeholder="Enter sub locality"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item>
            </Col>
            <Col xs={24} sm={24} md={12}>
              <Form.Item
                name="apartmentSociety"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    Apartment / Society (Optional)
                  </span>
                }
                style={{ fontFamily: "Poppins" }}
              >
                <Input
                  placeholder="Enter apartment or society name"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item>
            </Col>
          </Row>

          <Form.Item
            name="HouseNo"
            label={
              <span style={{ fontFamily: "Poppins" }}>House No (Optional)</span>
            }
            style={{ fontFamily: "Poppins" }}
          >
            <Input
              type="number"
              placeholder="Enter House No"
              style={{ fontFamily: "Poppins" }}
            />
          </Form.Item>
        </>
      ),
    },
    {
      title: <span style={{ fontFamily: "Poppins" }}>Property Profile</span>,
      icon: <InfoCircleOutlined />,
      content: (
        <>
          <Typography
            variant="h4"
            style={{ marginBottom: 6, fontWeight: 500, fontFamily: "Poppins" }}
          >
            <InfoCircleOutlined /> Property Profile
          </Typography>
          <Typography
            type="secondary"
            style={{
              fontSize: "14px",
              marginBottom: 16,
              fontFamily: "Poppins",
              color: "#778899",
            }}
          >
            Provide detailed information about your property features
          </Typography>

          <Divider orientation="left" style={{ fontFamily: "Poppins" }}>
            Room Details
          </Divider>

          <Row gutter={[16, 16]}>
            <Col xs={24} sm={24} md={8}>
              <Form.Item
                name="Bedrooms"
                label={
                  <span style={{ fontFamily: "Poppins" }}>No. of Bedrooms</span>
                }
                rules={[{ required: true, message: "Please enter Bedrooms" }]}
              >
                <Input
                  type="number"
                  placeholder="Enter No. Of Bedrooms"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item>
            </Col>
            <Col xs={24} sm={24} md={8}>
              <Form.Item
                name="Bathrooms"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    No. of Bathrooms
                  </span>
                }
                rules={[{ required: true, message: "Please enter Bathrooms" }]}
              >
                <Input
                  type="number"
                  placeholder="Enter No. of Bathrooms"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item>
            </Col>
            <Col xs={24} sm={24} md={8}>
              <Form.Item
                name="Balconies"
                label={<span style={{ fontFamily: "Poppins" }}>Balconies</span>}
              >
                <Input
                  type="number"
                  placeholder="Enter No. Balconies"
                  style={{ fontFamily: "Poppins" }}
                />
              </Form.Item>
            </Col>
          </Row>

          <Divider orientation="left" style={{ fontFamily: "Poppins" }}>
            Area Details
          </Divider>

          <Row gutter={[16, 16]}>
            <Col xs={24} sm={24} md={8}>
              <Form.Item
                name="supperbuilduparea"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    Plot Area (sq.ft)
                  </span>
                }
                rules={[
                  // { required: true, message: "Please enter Plot Area (sq.ft)" },
                  {
                    validator: (_, value) =>
                      !value || Number(value) > 0
                        ? Promise.resolve()
                        : Promise.reject(
                            new Error("Area must be a positive number")
                          ),
                  },
                ]}
              >
                <Input type="number" style={{ fontFamily: "Poppins" }} />
              </Form.Item>
            </Col>
            <Col xs={24} sm={24} md={8}>
              <Form.Item
                name="Buildup"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    Build-up Area (sq.ft)
                  </span>
                }
                rules={[
                  {
                    validator: (_, value) =>
                      !value || Number(value) > 0
                        ? Promise.resolve()
                        : Promise.reject(
                            new Error("Area must be a positive number")
                          ),
                  },
                ]}
              >
                <Input type="number" style={{ fontFamily: "Poppins" }} />
              </Form.Item>
            </Col>
            <Col xs={24} sm={24} md={8}>
              <Form.Item
                name="carpetarea"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    Carpet Area (sq.ft)
                  </span>
                }
                rules={[
                  {
                    validator: (_, value) =>
                      !value || Number(value) > 0
                        ? Promise.resolve()
                        : Promise.reject(
                            new Error("Area must be a positive number")
                          ),
                  },
                ]}
              >
                <Input type="number" style={{ fontFamily: "Poppins" }} />
              </Form.Item>
            </Col>
          </Row>

          <Form.Item
            name="totalfloor"
            label={<span style={{ fontFamily: "Poppins" }}>Total Floor</span>}
          >
            <Input
              type="number"
              placeholder="Enter Total Floor"
              style={{ fontFamily: "Poppins" }}
            />
          </Form.Item>

          <Form.Item
            label={<span style={{ fontFamily: "Poppins" }}>Furnishing</span>}
            style={{ fontFamily: "Poppins" }}
            required
          >
            <div style={{ display: "flex", flexWrap: "wrap", gap: "10px" }}>
              {furnishingTypes.map((item) => (
                <Button
                  key={item.id}
                  type={
                    selectedFurnishing === item.furnishingId
                      ? "primary"
                      : "default"
                  }
                  onClick={() => setSelectedFurnishing(item.furnishingId)}
                  style={{ fontFamily: "Poppins" }}
                >
                  {item.furnishingName}
                </Button>
              ))}
            </div>
          </Form.Item>
        </>
      ),
    },
    {
      title: <span style={{ fontFamily: "Poppins" }}>Photos and Videos</span>,
      icon: <FileImageOutlined />,
      content: (
        <>
          <Typography
            variant="h4"
            style={{ marginBottom: 6, fontWeight: 500, fontFamily: "Poppins" }}
          >
            <CameraOutlined /> Photos and Videos
          </Typography>
          <Typography
            type="secondary"
            style={{
              fontSize: "14px",
              marginBottom: 16,
              fontFamily: "Poppins",
              color: "#778899",
            }}
          >
            Add visual content to make your property more attractive
          </Typography>

          <Row gutter={[24, 24]}>
            <Col xs={24} md={12}>
              <Form.Item
                name="images"
                label={
                  <span style={{ fontFamily: "Poppins" }}>Upload Images</span>
                }
                // rules={[
                //   {
                //     validator: (_, value) =>
                //       fileList && fileList.length > 0
                //         ? Promise.resolve()
                //         : Promise.reject(
                //             new Error("Please upload at least one image")
                //           ),
                //   },
                // ]}
              >
                <Upload
                  listType="picture-card"
                  beforeUpload={() => false} // prevent automatic upload
                  fileList={fileList}
                  onChange={({ fileList: newFileList }) =>
                    setFileList(newFileList)
                  }
                  // multiple
                  maxCount={1}
                  accept="image/*"
                >
                  {/* <Button icon={<UploadOutlined />}>Click to Upload</Button> */}
                  {fileList.length >= 1 ? null : (
                    <div>
                      <PlusOutlined />
                      <div style={{ marginTop: 8, fontFamily: "Poppins" }}>
                        Upload
                      </div>
                    </div>
                  )}
                </Upload>
              </Form.Item>
            </Col>
            <Col xs={24} md={12}>
              {/* ✅ VIDEO UPLOAD */}
              <Form.Item
                name="video"
                label={
                  <span style={{ fontFamily: "Poppins" }}>
                    Upload Video (Optional)
                  </span>
                }
                style={{ fontFamily: "Poppins" }}
              >
                <Upload
                  beforeUpload={() => false}
                  fileList={videoFileList}
                  onChange={({ fileList }) => setVideoFileList(fileList)}
                  accept="video/*"
                  maxCount={1} // only 1 video allowed
                >
                  {/* <Button icon={<UploadOutlined />}>Upload Video</Button> */}
                  {videoFileList.length >= 1 ? null : (
                    <Button
                      icon={<VideoCameraOutlined />}
                      size="large"
                      style={{ fontFamily: "Poppins" }}
                    >
                      Upload Video
                    </Button>
                  )}
                </Upload>

                {/* ✅ SHOW EXISTING VIDEO IF EDIT MODE */}
                {existingVideo && (
                  <video
                    src={existingVideo}
                    controls
                    style={{
                      width: "100%",
                      marginTop: 10,
                      borderRadius: 8,
                      border: "1px solid #ddd",
                      padding: 10,
                    }}
                  />
                )}
              </Form.Item>
            </Col>
          </Row>

          <Form.Item
            name="description"
            label={<span style={{ fontFamily: "Poppins" }}>Description</span>}
          >
            <Input.TextArea
              rows={4}
              placeholder="Provide a detailed description of your property..."
              style={{ fontFamily: "Poppins" }}
            />
          </Form.Item>
        </>
      ),
    },
    {
      title: (
        <span style={{ fontFamily: "Poppins" }}>Pricing, Other and Submit</span>
      ),
      icon: <DollarOutlined />,
      content: (
        <>
          <Typography
            variant="h4"
            style={{ marginBottom: 6, fontWeight: 500, fontFamily: "Poppins" }}
          >
            <DollarOutlined /> Pricing, Other and Submit
          </Typography>
          <Typography
            type="secondary"
            style={{
              fontSize: "14px",
              marginBottom: 16,
              fontFamily: "Poppins",
              color: "#778899",
            }}
          >
            Set the right price to attract potential buyers
          </Typography>

          <Row gutter={[24, 24]}>
            <Col xs={24} md={12}>
              {/* ✅ PRICE FIELD */}
              <Form.Item
                name="price"
                label={<span style={{ fontFamily: "Poppins" }}>Price</span>}
                rules={[
                  { required: true, message: "Please enter price" },
                  {
                    type: "number",
                    min: 0,
                    transform: (value) => Number(value),
                    message: "Price must be a positive number",
                  },
                ]}
              >
                <Input
                  type="number"
                  prefix="₹"
                  style={{ fontFamily: "Poppins" }}
                  onChange={(e) => {
                    const price = Number(e.target.value);
                    const plotArea = Number(
                      form.getFieldValue("supperbuilduparea") || 0
                    );

                    if (plotArea > 0) {
                      const pricepersqft = price / plotArea;
                      form.setFieldsValue({
                        pricepersqft: pricepersqft.toFixed(2),
                      });
                    }
                  }}
                />
              </Form.Item>
            </Col>
            <Col xs={24} md={12}>
              {/* ✅ PRICE PER SQ. FT */}
              <Form.Item
                label={
                  <span style={{ fontFamily: "Poppins" }}>Price per sq.ft</span>
                }
                name="pricepersqft"
              >
                <Input
                  style={{ fontFamily: "Poppins" }}
                  readOnly
                  placeholder="Auto Calculated"
                  prefix="₹"
                />
              </Form.Item>
            </Col>
          </Row>
          <Divider />

          <div
            className="submit-section"
            style={{
              marginTop: "24px",
              border: "1px solid #778899",
              padding: 12,
              borderRadius: 8,
              marginBottom: 20,
            }}
          >
            <Typography
              level={4}
              style={{
                fontSize: "18px",
                fontWeight: 500,
                fontFamily: "Poppins",
                marginLeft: 10,
              }}
            >
              Review Your Property Details
            </Typography>
            <Typography
              type="secondary"
              style={{
                fontSize: "14px",
                fontFamily: "Poppins",
                color: "#778899",
                marginLeft: 10,
              }}
            >
              Please review all the information before submitting
            </Typography>

            <div
              className="property-summary"
              style={{
                backgroundColor: "#f5f7fa",
                padding: "20px",
                borderRadius: "8px",
                marginTop: "16px",
              }}
            >
              <Row gutter={[16, 16]}>
                <Col xs={24} sm={12}>
                  <div className="summary-item" style={{ marginBottom: "8px" }}>
                    <Typography
                      strong
                      style={{ fontSize: 18, fontFamily: "Poppins" }}
                    >
                      Property Type :
                    </Typography>
                    <div
                      style={{
                        fontSize: 16,
                        fontFamily: "Poppins",
                        color: "#778899",
                      }}
                    >
                      {selectedPostpeopertyTypes &&
                        PostpeopertyTypes.find(
                          (t) => t.postTypeId === selectedPostpeopertyTypes
                        )?.posttypeName}
                      {" - "}
                      {selectedPropertyType &&
                        propertyTypes.find(
                          (t) => t.propertyTypeId === selectedPropertyType
                        )?.propertytypeName}
                    </div>
                  </div>
                </Col>
                <Col xs={24} sm={12}>
                  <div className="summary-item" style={{ marginBottom: "8px" }}>
                    <Typography
                      strong
                      style={{ fontSize: 18, fontFamily: "Poppins" }}
                    >
                      Area :
                    </Typography>
                    <div
                      style={{
                        fontSize: 16,
                        fontFamily: "Poppins",
                        color: "#778899",
                      }}
                    >
                      {form.getFieldValue("supperbuilduparea")} sq.ft
                    </div>
                  </div>
                </Col>
                <Col xs={24} sm={12}>
                  <div className="summary-item" style={{ marginBottom: "8px" }}>
                    <Typography
                      strong
                      style={{ fontSize: 18, fontFamily: "Poppins" }}
                    >
                      Bedrooms :
                    </Typography>
                    <div
                      style={{
                        fontSize: 16,
                        fontFamily: "Poppins",
                        color: "#778899",
                      }}
                    >
                      {form.getFieldValue("Bedrooms")}
                    </div>
                  </div>
                </Col>
                <Col xs={24} sm={12}>
                  <div className="summary-item" style={{ marginBottom: "8px" }}>
                    <Typography
                      strong
                      style={{ fontSize: 18, fontFamily: "Poppins" }}
                    >
                      Bathrooms :
                    </Typography>
                    <div
                      style={{
                        fontSize: 16,
                        fontFamily: "Poppins",
                        color: "#778899",
                      }}
                    >
                      {form.getFieldValue("Bathrooms")}
                    </div>
                  </div>
                </Col>
              </Row>
            </div>
          </div>
        </>
      ),
    },
  ];

  return (
    <>
      {/* Header */}
      <Box
        sx={{
          px: { xs: 1.5, sm: 1 },
          pt: 1.3,
          minHeight: "10vh",
        }}
      >
        <Paper
          elevation={4}
          sx={{
            p: { xs: 2, sm: 2 },
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 50,
          }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
            }}
          >
            {/* Left side title */}
            <Typography
              variant="h6"
              fontWeight={700}
              sx={{
                color: "#1f2937",
                fontSize: { xs: "1.1rem", sm: "1.45rem" },
                display: "flex",
                alignItems: "center",
                gap: 1,
                fontFamily: "Poppins",
              }}
            >
              Post Your Property
            </Typography>

            {/* Right side AntD Dashboard Progress */}
            <Progress
              type="dashboard"
              percent={Math.round((current / (steps.length - 1)) * 100)}
              width={80}
              size={50}
              strokeWidth={15}
              steps={5}
              trailColor="rgba(0, 0, 0, 0.06)"
            />
          </Box>
        </Paper>
      </Box>

      {/* Body */}
      <Box
        sx={{
          px: { xs: 1.5, sm: 1 },
          pt: 1.3,
        }}
      >
        <Paper
          elevation={4}
          sx={{
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 90,
            p: 1,
            px: 2,
          }}
        >
          {/* Stepper */}
          <Steps current={current} style={{ margin: "20px 0px" }}>
            {steps.map((item, index) => (
              <Step
                key={item.title}
                title={item.title}
                icon={
                  <div
                    style={{
                      fontSize: 20,
                      color: current >= index ? "#1890ff" : "#d9d9d9",
                    }}
                  >
                    {item.icon}
                  </div>
                }
              />
            ))}
          </Steps>
        </Paper>
      </Box>

      <Box
        sx={{
          px: { xs: 1.5, sm: 1.3 },
          pt: 1,
          minHeight: "90vh",
        }}
      >
        <Paper
          elevation={4}
          sx={{
            borderRadius: 4,
            backdropFilter: "blur(8px)",
            background: "white",
            boxShadow: "0 8px 24px rgba(0,0,0,0.05)",
            minHeight: 550,
            p: 3,
          }}
        >
          {/* Form with Ads */}
          <Row justify="center" gutter={[24, 24]}>
            <Col xs={24} sm={24} md={14} lg={16}>
              <Form
                form={form}
                layout="vertical"
                onFinish={current === steps.length - 1 ? onFinish : undefined}
              >
                {steps[current].content}

                <Form.Item>
                  <Box
                    sx={{
                      display: "flex",
                      justifyContent: "end",
                      width: "100%",
                    }}
                  >
                    {current > 0 && (
                      <Button
                        type="primary"
                        onClick={prev}
                        style={{
                          padding: "8px 28px",
                          minWidth: "120px",
                          fontWeight: 600,
                          borderRadius: "2px",
                          boxShadow: "0 4px 8px rgba(0,0,0,0.15)",
                          transition: "all 0.3s ease",
                          fontFamily: "Poppins",
                        }}
                        onMouseOver={(e) => {
                          e.currentTarget.style.transform = "translateY(-2px)";
                          e.currentTarget.style.boxShadow =
                            "0 6px 12px rgba(0,0,0,0.2)";
                        }}
                        onMouseOut={(e) => {
                          e.currentTarget.style.transform = "translateY(0)";
                          e.currentTarget.style.boxShadow =
                            "0 4px 8px rgba(0,0,0,0.15)";
                        }}
                      >
                        ← Previous
                      </Button>
                    )}
                    {current < steps.length - 1 && (
                      <Button
                        type="primary"
                        onClick={next}
                        disabled={
                          current === 0 &&
                          (!selectedmainPropertyTypes ||
                            !selectedPostpeopertyTypes ||
                            !selectedPropertyType)
                        }
                        style={{
                          padding: "8px 28px",
                          minWidth: "120px",
                          fontWeight: 600,
                          borderRadius: "2px",
                          boxShadow: "0 4px 8px rgba(0,0,0,0.15)",
                          transition: "all 0.3s ease",
                          marginLeft: "10px",
                          fontFamily: "Poppins",
                        }}
                        onMouseOver={(e) => {
                          e.currentTarget.style.transform = "translateY(-2px)";
                          e.currentTarget.style.boxShadow =
                            "0 6px 12px rgba(0,0,0,0.2)";
                        }}
                        onMouseOut={(e) => {
                          e.currentTarget.style.transform = "translateY(0)";
                          e.currentTarget.style.boxShadow =
                            "0 4px 8px rgba(0,0,0,0.15)";
                        }}
                      >
                        Next →
                      </Button>
                    )}
                    {current === steps.length - 1 && (
                      <Button
                        type="primary"
                        htmlType="submit"
                        icon={<CheckCircleOutlined />}
                        style={{
                          padding: "10px 24px",
                          minWidth: "150px",
                          fontWeight: 600,
                          borderRadius: "2px",
                          background:
                            "linear-gradient(90deg, #28a745, #34d058)",
                          border: "none",
                          boxShadow: "0 4px 10px rgba(40, 167, 69, 0.3)",
                          display: "flex",
                          alignItems: "center",
                          justifyContent: "center",
                          gap: "8px",
                          transition: "all 0.3s ease",
                          fontFamily: "Poppins",
                          marginLeft: "10px",
                        }}
                        onMouseEnter={(e) => {
                          e.currentTarget.style.background =
                            "linear-gradient(90deg, #34d058, #50fa7b)";
                          e.currentTarget.style.boxShadow =
                            "0 6px 14px rgba(40, 167, 69, 0.4)";
                        }}
                        onMouseLeave={(e) => {
                          e.currentTarget.style.background =
                            "linear-gradient(90deg, #28a745, #34d058)";
                          e.currentTarget.style.boxShadow =
                            "0 4px 10px rgba(40, 167, 69, 0.3)";
                        }}
                      >
                        {postpropertyId ? "Update" : "Submit"}
                      </Button>
                    )}
                  </Box>
                </Form.Item>
              </Form>
            </Col>

            <Col
              xs={24}
              sm={24}
              md={10}
              lg={8}
              style={{
                // background: "rgb(245, 245, 245)",
                display: "flex",
                justifyContent: "center",
                alignItems: "center",
                paddingBottom: "10px",
              }}
            >
              <img
                src={Ads}
                alt="Property Banner"
                style={{
                  maxWidth: "100%",
                  height: "600px",
                  borderRadius: "8px",
                }}
              />
            </Col>
          </Row>
        </Paper>
      </Box>
    </>
  );
};

export default PropertyForm3;
