import React, { useState, useEffect } from "react";
import { Row, Col, Button, Rate } from "antd";
import axios from "axios";
import { URL } from "../../config";
import Img1 from "../../assets/PostIMG/PostImg_11.jfif";
import Img2 from "../../assets/PostIMG/PostImg_12.avif";
import Img3 from "../../assets/PostIMG/PostImg_13.avif";

const NewLaunch = () => {
  const [isMobile, setIsMobile] = useState(false);
  const [centerIdx, setCenterIdx] = useState(1);
  const [properties, setProperties] = useState([]);

  useEffect(() => {
    const handleResize = () => setIsMobile(window.innerWidth <= 768);
    handleResize();
    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, []);

  useEffect(() => {
    const interval = setInterval(() => {
      setCenterIdx((prev) => (prev + 1) % images.length);
    }, 2500);
    return () => clearInterval(interval);
  }, []);

  // 🌐 Fetch New Launch data
  useEffect(() => {
    const fetchNewLaunch = async () => {
      try {
        const res = await axios.get(`${URL}/post-property-masters/new-launch`);
        setProperties(res.data || []);
      } catch (err) {
        console.error("Error fetching new launch:", err);
      }
    };
    fetchNewLaunch();
  }, []);

  const sectionStyle = {
    background: "#ffff",
    width: "100%",
    minHeight: "85vh",
    display: "flex",
    alignItems: "center",
    justifyContent: "center",
    padding: "20px 30px",
    overflow: "hidden",
  };

  const jointBoxStyle = {
    background: "rgba(234, 246, 255, 0.85)",
    borderRadius: "30px",
    display: "flex",
    flexWrap: "wrap",
    alignItems: "center",
    justifyContent: "space-between",
    padding: "30px 40px",
    boxShadow: "0 4px 20px rgba(0,0,0,0.15)",
    width: "100%",
    maxWidth: "1300px",
  };

  const propertyCard = {
    borderBottom: "1px solid #ddddddff",
    padding: "16px 0",
    display: "flex",
    justifyContent: "space-between",
    alignItems: "center",
  };

  const rightImagesContainer = {
    position: "relative",
    width: "100%",
    height: isMobile ? "280px" : "420px",
    display: "flex",
    alignItems: "center",
    justifyContent: "center",
  };

  const rightImageStyle = (pos) => {
    const baseWidth = isMobile ? 90 : 230;
    const centerWidth = isMobile ? 120 : 250;

    let width = pos === 1 ? centerWidth : baseWidth;
    let height = width * 1.7;
    let zIndex = pos === 1 ? 2 : 1;
    let offset = pos === 0 ? -width / 1.3 : pos === 1 ? 0 : width / 1.3;

    return {
      position: "absolute",
      width: `${width}px`,
      height: `${height}px`,
      objectFit: "cover",
      borderRadius: "50% / 15%",
      border: "4px solid white",
      boxShadow: "0 4px 15px rgba(0,0,0,0.2)",
      transform: `translateX(${offset}px)`,
      zIndex: zIndex,
      transition: "all 0.8s cubic-bezier(.77,0,.175,1)",
      backgroundColor: "#eee",
      padding: "0%"
    };
  };

  const images = [Img1, Img2, Img3];

  const getImgDisplayOrder = () => [
    (centerIdx + images.length - 1) % images.length,
    centerIdx,
    (centerIdx + 1) % images.length,
  ];

  const formatPriceINR = (price) => {
    if (!price || isNaN(price)) return "On Request";

    if (price >= 10000000) {
      return (price / 10000000).toFixed(2).replace(/\.00$/, "") + " Cr";
    }

    if (price >= 100000) {
      return (price / 100000).toFixed(2).replace(/\.00$/, "") + " Lakh";
    }

    return price.toLocaleString("en-IN");
  };

   const getImage = (item) => {
    if (!item?.photo) return Img1;
    if (item.photo.includes("default.jpg")) return Img1;
    return item.photo;
  };

  return (
    <div style={sectionStyle}>
      <div style={jointBoxStyle}>
        <Row gutter={[10, 10]} align="middle" style={{ width: "100%" }}>
          {/* LEFT CONTENT */}
          <Col xs={24} sm={24} md={12}>
            <h1
              style={{
                color: "#1d4ed8",
                fontWeight: 600,
                fontSize: isMobile ? "28px" : "36px",
                marginBottom: "10px",
                fontFamily: "Poppins",
              }}
            >
              New Launch
            </h1>

            <Rate disabled defaultValue={5} style={{ color: "#facc15" }} />

            <p
              style={{
                marginTop: 20,
                color: "#333",
                lineHeight: 1.6,
                fontSize: isMobile ? "14px" : "16px",
                fontFamily: "Poppins",
              }}
            >
              Design, Develop, Manufacture, and Supply of Valves and Allied
              Products to the total satisfaction of its customers through
              innovation, world-class leadership, employee development, and
              social responsibility;
            </p>

            {properties.slice(0, 3).map((item) => (
              <div key={item.postpropertyId} style={propertyCard}>
                <div>
                  <h3
                    style={{
                      marginBottom: 5,
                      color: "#000",
                      fontSize: isMobile ? "16px" : "16px",
                      fontFamily: "Poppins",
                    }}
                  >
                    {item.Property_Title || "Property"}
                  </h3>
                  <p
                    style={{
                      margin: 0,
                      color: "#666",
                      fontSize: isMobile ? "12px" : "13px",
                      fontFamily: "Poppins",
                    }}
                  >
                    {item.propertyType?.propertytypeName || "Property"} •{" "}
                    {item.postType?.posttypeName || "Sale"}
                  </p>
                  <p
                    style={{
                      margin: "5px 0 0",
                      fontSize: isMobile ? "12px" : "13px",
                      color: "#888",
                      fontFamily: "Poppins",
                    }}
                  >
                     {item.fkcityId ?? "-"} ||
                    {item.fklocalityId ?? "-"}
                  </p>
                </div>
                <Button
                  type="primary"
                  shape="round"
                  style={{
                    background: "#002b1e",
                    border: "none",
                    fontWeight: 500,
                    fontSize: isMobile ? "12px" : "14px",
                    fontFamily: "Poppins",
                  }}
                >
                  ₹ {formatPriceINR(item.price)}
                </Button>
              </div>
            ))}
          </Col>

          <Col xs={24} sm={24} md={12}>
            <div style={rightImagesContainer}>
               {getImgDisplayOrder().map((idx, pos) => (
                <img
                  key={idx}
                  src={getImage(properties[idx])}
                  alt="Property"
                  style={rightImageStyle(pos)}
                />
              ))}
            </div>
          </Col>
        </Row>
      </div>
    </div>
  );
};

export default NewLaunch;
