import React, { useState, useEffect } from "react";
import {
  AppBar,
  Toolbar,
  Box,
  useMediaQuery,
  useTheme,
  Menu,
  MenuItem,
  IconButton,
  Avatar,
  ListItemIcon,
  ListItemText,
  Divider,
  Typography,
  Chip,
} from "@mui/material";
import { Crown } from "lucide-react";
import ShoppingCartIcon from "@mui/icons-material/ShoppingCart";
import PersonSharpIcon from "@mui/icons-material/PersonSharp";
import DashboardIcon from "@mui/icons-material/Dashboard";
import LogoutIcon from "@mui/icons-material/Logout";
import VisibilityIcon from "@mui/icons-material/Visibility";
import DesktopNav from "./DesktopNav";
import MobileNav from "./MobileNav";
import clinicLogo from "../../../assets/Logo/Logo-lello2.jfif";
import GoogleTranslate from "../../../assets/Header/GoogleTranslate1.png";
import { useNavigate } from "react-router-dom";
import "./Header.css";
import { Button, Badge } from "antd";

const navLinks = [
  { label: "Home", path: "/" },
  { label: "About us", path: "/about-us" },
  { label: "Buy" },
  { label: "Sell" },
  { label: "Rent" },
  { label: "Contact us", path: "/contact-us" },
];

const Header = ({ onLoginClick, user }) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("md"));
  const [language, setLanguage] = useState("en");
  const [anchorEl, setAnchorEl] = useState(null);
  const [profileAnchorEl, setProfileAnchorEl] = useState(null);
  const [isTranslateReady, setIsTranslateReady] = useState(false);
  const navigate = useNavigate();

  // 🔹 Check login status
  const sellerId = localStorage.getItem("sellerId");
  const buyerId = localStorage.getItem("buyerId");

  const isLoggedIn =
    !!user ||
    (sellerId && sellerId !== "null" && sellerId !== "undefined") ||
    (buyerId && buyerId !== "null" && buyerId !== "undefined");

  // 🔹 Load Google Translate Script
  useEffect(() => {
    if (window.google && window.google.translate) return;

    const script = document.createElement("script");
    script.src =
      "https:/translate.google.com/translate_a/element.js?cb=googleTranslateElementInit";
    script.async = true;
    document.body.appendChild(script);

    window.googleTranslateInit = () => {
      new window.google.translate.TranslateElement(
        {
          pageLanguage: "en",
          includedLanguages: "en,hi,mr",
          autoDisplay: false,
        },
        "google_translate_element"
      );
      setIsTranslateReady(true);
    };
  }, []);

  // 🔹 Handle Language Change
  const handleLanguageChange = (lang) => {
    setLanguage(lang);
    localStorage.setItem("selectedLanguage", lang);

    const interval = setInterval(() => {
      const selectBox = document.querySelector(".goog-te-combo");
      if (selectBox) {
        selectBox.value = lang;
        selectBox.dispatchEvent(new Event("change"));
        clearInterval(interval);
      }
    }, 100);
  };

  // 🔹 Restore saved language on page load
  useEffect(() => {
    if (!isTranslateReady) return;

    const savedLang = localStorage.getItem("selectedLanguage");
    if (!savedLang || savedLang === "en") return;

    const interval = setInterval(() => {
      const selectBox = document.querySelector(".goog-te-combo");
      if (selectBox) {
        selectBox.value = savedLang;
        selectBox.dispatchEvent(new Event("change"));
        setLanguage(savedLang);
        clearInterval(interval);
      }
    }, 200);
  }, [isTranslateReady]);

  const handleToggleMenu = (event) => {
    setAnchorEl(anchorEl ? null : event.currentTarget);
  };

  const handleCloseMenu = () => {
    setAnchorEl(null);
  };

  const handleProfileMenuOpen = (event) => {
    setProfileAnchorEl(event.currentTarget);
  };

  const handleProfileMenuClose = () => {
    setProfileAnchorEl(null);
  };

  const handleLogout = () => {
    localStorage.removeItem("sellerId");
    localStorage.removeItem("buyerId");
    localStorage.removeItem("user");
    localStorage.removeItem("selectedLanguage");
    localStorage.removeItem("role");
    localStorage.removeItem("id");
    localStorage.removeItem("userTypeId");
    handleProfileMenuClose();
    window.location.href = "/";
  };

  const handleDashboard = () => {
    navigate("/dashboard");
    handleProfileMenuClose();
  };

  const handleMembership = () => {
    navigate("/membership");
    handleProfileMenuClose();
  };

  const handleAddToCart = () => {
    navigate("/addtocart");
    handleProfileMenuClose();
  };

    const handleViewsProperties = () => {
    navigate("/view-properties");
    handleProfileMenuClose();
  };


  // 🔹 Account / Login Button logic with MUI Design
  const renderLoginButton = () => {
    if (isLoggedIn) {
      const isSeller =
        sellerId && sellerId !== "null" && sellerId !== "undefined";

      const userName = user?.buyerName || user?.sellerName || "User";
      const userInitial = userName.charAt(0).toUpperCase();

      return (
        <>
          <IconButton
            onClick={handleProfileMenuOpen}
            sx={{
              p: 0,
              transition: "all 0.3s ease",
              "&:hover": {
                transform: "scale(1.05)",
              },
            }}
          >
            <Avatar
              sx={{
                width: 45,
                height: 45,
                bgcolor: "#51aed6",
                border: "2px solid white",
                fontWeight: 600,
                fontSize: "1.1rem",
                boxShadow: "0 4px 12px rgba(0,0,0,0.15)",
                transition: "all 0.3s ease",
                "&:hover": {
                  boxShadow: "0 6px 16px rgba(0,0,0,0.25)",
                },
              }}
            >
              {userInitial}
            </Avatar>
          </IconButton>

          <Menu
            anchorEl={profileAnchorEl}
            open={Boolean(profileAnchorEl)}
            onClose={handleProfileMenuClose}
            onClick={handleProfileMenuClose}
            transformOrigin={{ horizontal: "right", vertical: "top" }}
            anchorOrigin={{ horizontal: "right", vertical: "bottom" }}
            PaperProps={{
              elevation: 8,
              sx: {
                mt: 1.5,
                minWidth: 220,
                borderRadius: "12px",
                overflow: "visible",
                boxShadow: "0 8px 24px rgba(0,0,0,0.12)",
                "&:before": {
                  content: '""',
                  display: "block",
                  position: "absolute",
                  top: 0,
                  right: 20,
                  width: 10,
                  height: 10,
                  bgcolor: "background.paper",
                  transform: "translateY(-50%) rotate(45deg)",
                  zIndex: 0,
                },
              },
            }}
          >
            <Box sx={{ px: 2.5, py: 2, textAlign: "center" }}>
              <Avatar
                sx={{
                  width: 60,
                  height: 60,
                  bgcolor: "#4caf50",
                  margin: "0 auto",
                  mb: 1,
                  fontWeight: 700,
                  fontSize: "1.5rem",
                }}
              >
                {userInitial}
              </Avatar>
              <Typography
                variant="subtitle1"
                sx={{ fontFamily: "Poppins" }}
                fontWeight={600}
                color="text.primary"
              >
                {userName}
              </Typography>
              {isSeller && (
                <Chip
                  label="Seller"
                  size="small"
                  color="success"
                  sx={{ mt: 0.5, fontSize: "0.7rem", height: 20 }}
                />
              )}
            </Box>

            <Divider />

            {isSeller && (
              <MenuItem
                onClick={handleDashboard}
                sx={{
                  py: 1.5,
                  px: 2.5,
                  "&:hover": {
                    bgcolor: "#f5f5f5",
                  },
                }}
              >
                <ListItemIcon>
                  <DashboardIcon fontSize="small" sx={{ color: "#1976d2" }} />
                </ListItemIcon>
                <ListItemText>
                  <Typography
                    variant="body2"
                    sx={{ fontFamily: "Poppins" }}
                    fontWeight={500}
                  >
                    Dashboard
                  </Typography>
                </ListItemText>
              </MenuItem>
            )}
            <MenuItem
              onClick={handleMembership}
              sx={{
                py: 1.5,
                px: 2.5,
                "&:hover": {
                  bgcolor: "#f5f5f5",
                },
              }}
            >
              <ListItemIcon>
                <Crown color="#FFD700" fill="#FFD700" />
              </ListItemIcon>
              <ListItemText>
                <Typography
                  variant="body2"
                  sx={{ fontFamily: "Poppins" }}
                  fontWeight={500}
                >
                  Membership
                </Typography>
              </ListItemText>
            </MenuItem>
             <MenuItem
              onClick={handleAddToCart}
              sx={{
                py: 1.5,
                px: 2.5,
                "&:hover": {
                  bgcolor: "#f5f5f5",
                },
              }}
            >
              <ListItemIcon>
                 <ShoppingCartIcon sx={{ color: "#51aed6" }} />
              </ListItemIcon>
              <ListItemText>
                <Typography
                  variant="body2"
                  sx={{ fontFamily: "Poppins" }}
                  fontWeight={500}
                >
                  Add To Cart
                </Typography>
              </ListItemText>
            </MenuItem>
             <MenuItem
              onClick={handleViewsProperties}
              sx={{
                py: 1.5,
                px: 2.5,
                "&:hover": {
                  bgcolor: "#f5f5f5",
                },
              }}
            >
              <ListItemIcon>
                 <VisibilityIcon sx={{ color: "#51aed6" }} />
              </ListItemIcon>
              <ListItemText>
                <Typography
                  variant="body2"
                  sx={{ fontFamily: "Poppins" }}
                  fontWeight={500}
                >
                  Views Properties
                </Typography>
              </ListItemText>
            </MenuItem>
            <MenuItem
              onClick={handleLogout}
              sx={{
                py: 1.5,
                px: 2.5,
                "&:hover": {
                  bgcolor: "#ffebee",
                },
              }}
            >
              <ListItemIcon>
                <LogoutIcon fontSize="small" sx={{ color: "#d32f2f" }} />
              </ListItemIcon>
              <ListItemText>
                <Typography
                  variant="body2"
                  sx={{ fontFamily: "Poppins" }}
                  fontWeight={500}
                  color="#d32f2f"
                >
                  Logout
                </Typography>
              </ListItemText>
            </MenuItem>
          </Menu>
        </>
      );
    } else {
      return (
        <Box sx={{ display: "inline-block" }}>
          <Badge.Ribbon
            size="small"
            text="FREE"
            color="green"
            style={{ marginTop: -8 }}
          >
            <Button
              type="primary"
              icon={<PersonSharpIcon />}
              onClick={onLoginClick}
              style={{
                marginTop: 10,
                padding: "20px 10px 20px 10px",
                fontWeight: 500,
                borderRadius: "8px",
                borderColor: "#255670",
                boxShadow: "0 4px 12px rgba(0,0,0,0.15)",
                fontFamily: "Poppins",
              }}
            >
              Login / Signup
            </Button>
          </Badge.Ribbon>
        </Box>
      );
    }
  };

  return (
    <>
      {/* Hidden Google Translate widget */}
      <div
        id="google_translate_element"
        style={{
          position: "fixed",
          top: "-9999px",
          left: "-9999px",
          visibility: "hidden",
          opacity: 0,
          pointerEvents: "none",
        }}
      ></div>

      <AppBar
        position="fixed"
        sx={{
          background: "#255670",
          boxShadow: "0px 4px 10px rgba(0, 0, 0, 0.15)",
          zIndex: 999,
        }}
      >
        <Toolbar
          sx={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            px: { xs: 2, md: 6 },
            py: 0.5,
          }}
        >
          {/* 🔹 Left Logo */}
          <Box
            sx={{
              display: "flex",
              alignItems: "center",
              backgroundColor: "#f4b62e",
              borderRadius: "6px",
              boxShadow: "2px 2px 8px rgba(0,0,0,0.15)",
              clipPath: "polygon(0 0, 100% 0, 80% 100%, 0 100%)",
              width: { xs: "150px", sm: "200px", md: "300px" },
              height: { xs: "55px", md: "60px" },
              cursor: "pointer",
            }}
            onClick={() => navigate("/")}
          >
            <img
              src={clinicLogo}
              alt="Clinic Logo"
              className="clinic-logo"
              style={{ height: "80px", width: "100%" }}
            />
          </Box>

          {/* 🔹 Navigation Links */}
          {isMobile ? (
            <MobileNav navLinks={navLinks} />
          ) : (
            <DesktopNav navLinks={navLinks} />
          )}

          {/* 🔹 Right Side */}
          {!isMobile && (
            <Box
              sx={{
                display: "flex",
                alignItems: "center",
                color: "white",
                gap: 3,
              }}
            >
              {/* 🌐 Google Translate */}
              <Box sx={{ position: "relative" }}>
                <IconButton
                  onClick={handleToggleMenu}
                  sx={{
                    p: 0,
                    width: 50,
                    height: 50,
                    borderRadius: "50%",
                    "&:hover": {
                      backgroundColor: "#255670e0",
                      border: "2px solid white",
                    },
                  }}
                >
                  <img
                    src={GoogleTranslate}
                    alt="Google Translate"
                    style={{ width: "40px", height: "30px", marginTop: "10px" }}
                  />
                </IconButton>

                <Menu
                  anchorEl={anchorEl}
                  open={Boolean(anchorEl)}
                  onClose={handleCloseMenu}
                  anchorOrigin={{
                    vertical: "bottom",
                    horizontal: "center",
                  }}
                  transformOrigin={{
                    vertical: "top",
                    horizontal: "center",
                  }}
                  PaperProps={{
                    sx: {
                      mt: 1,
                      backgroundColor: "#fff",
                      borderRadius: "8px",
                      boxShadow: "0px 4px 12px rgba(0,0,0,0.15)",
                    },
                  }}
                >
                  <MenuItem
                    onClick={() => {
                      handleLanguageChange("en");
                      handleCloseMenu();
                    }}
                    selected={language === "en"}
                  >
                    <Box
                      sx={{
                        display: "flex",
                        alignItems: "center",
                        gap: 1,
                        fontFamily: "Poppins",
                      }}
                    >
                      <span>🇬🇧</span>
                      <span>English</span>
                      {language === "en" && (
                        <Chip
                          label="Active"
                          size="small"
                          color="primary"
                          sx={{ ml: 1 }}
                        />
                      )}
                    </Box>
                  </MenuItem>
                  <MenuItem
                    onClick={() => {
                      handleLanguageChange("hi");
                      handleCloseMenu();
                    }}
                    selected={language === "hi"}
                  >
                    <Box
                      sx={{
                        display: "flex",
                        alignItems: "center",
                        gap: 1,
                        fontFamily: "Poppins",
                      }}
                    >
                      <span>🇮🇳</span>
                      <span>Hindi</span>
                      {language === "hi" && (
                        <Chip
                          label="Active"
                          size="small"
                          color="primary"
                          sx={{ ml: 1 }}
                        />
                      )}
                    </Box>
                  </MenuItem>
                  <MenuItem
                    onClick={() => {
                      handleLanguageChange("mr");
                      handleCloseMenu();
                    }}
                    selected={language === "mr"}
                  >
                    <Box
                      sx={{
                        display: "flex",
                        alignItems: "center",
                        gap: 1,
                        fontFamily: "Poppins",
                      }}
                    >
                      <span>🇮🇳</span>
                      <span>Marathi</span>
                      {language === "mr" && (
                        <Chip
                          label="Active"
                          size="small"
                          color="primary"
                          sx={{ ml: 1 }}
                        />
                      )}
                    </Box>
                  </MenuItem>
                </Menu>
              </Box>

              {/* 👤 Login / Account */}
              {renderLoginButton()}
            </Box>
          )}
        </Toolbar>
      </AppBar>

      <style>{`
        /* Hide Google Translate elements */
        .goog-te-banner-frame.skiptranslate { 
          display: none !important; 
        }
        body { 
          top: 0 !important; 
        }
        .goog-te-balloon-frame, 
        .goog-logo-link, 
        .goog-te-gadget, 
        iframe.skiptranslate { 
          display: none !important; 
        }
        
        /* Hide the Google Translate toolbar */
        .goog-te-banner-frame {
          display: none !important;
        }
        
        /* Remove extra padding added by Google Translate */
        body {
          position: static !important;
          top: 0 !important;
        }
        
        /* Style adjustments for translated content */
        .translated-ltr {
          margin-top: 0 !important;
        }
        
        /* Hide the Google branding */
        .goog-logo-link {
          display: none !important;
        }
        
        .goog-te-gadget {
          color: transparent !important;
        }
        
        /* Additional fixes */
        #goog-gt-tt {
          display: none !important;
        }
        
        .goog-te-spinner-pos {
          display: none !important;
        }
      `}</style>
    </>
  );
};

export default Header;
