import React, { useEffect, useState } from "react";
import { Row, Col, Typography, Tag, Button } from "antd";
import { LeftOutlined, RightOutlined } from "@ant-design/icons";
import { motion } from "framer-motion";
import { useNavigate } from "react-router-dom";
import Slider from "react-slick";
import axios from "axios";
import "./blogsection.css";
import { URL } from "../../../config";

const { Title, Text } = Typography;

const PrevArrow = ({ onClick }) => (
  <div className="blog-arrow blog-arrow-left" onClick={onClick}>
    <LeftOutlined />
  </div>
);

const NextArrow = ({ onClick }) => (
  <div className="blog-arrow blog-arrow-right" onClick={onClick}>
    <RightOutlined />
  </div>
);

const settings = {
  autoplay: true,
  autoplaySpeed: 5000,
  speed: 1300,
  dots: false,
  infinite: true,
  slidesToScroll: 1,
  slidesToShow: 3,
  prevArrow: <PrevArrow />,
  nextArrow: <NextArrow />,
  responsive: [
    { breakpoint: 992, settings: { slidesToShow: 2 } },
    { breakpoint: 576, settings: { slidesToShow: 1 } },
  ],
};

const BlogSection = () => {
  const navigate = useNavigate();
  const [blogs, setBlogs] = useState([]);
  const [expanded, setExpanded] = useState({});

  useEffect(() => {
    axios
      .get(`${URL}/real-estate-masters`)
      .then((res) => setBlogs(res.data || []))
      .catch(console.error);
  }, []);

  const toggleReadMore = (id) => {
    setExpanded((prev) => ({ ...prev, [id]: !prev[id] }));
  };

  return (
    <section className="blog-wrapper">
      <div className="blog-header">
        <Title level={1} className="blog-title">
          Stay informed with our latest
          <br />
          real estate insights
        </Title>

        <Button type="primary" className="blog-cta">
          Browse all Blog
        </Button>
      </div>

      <Slider {...settings}>
        {blogs.map((post) => {
          const isExpanded = expanded[post.blogId];

          return (
            <div key={post.blogId}>
              <Row justify="center">
                <Col xs={24} md={22} lg={22}>
                  <motion.article
                    className={`blog-card ${isExpanded ? "expanded" : ""}`}
                    whileHover={{ scale: 1.04 }}
                    onClick={() => navigate(`/blog/${post.blogId}`)}
                  >
                    <div className="blog-card-image-wrap">
                      <motion.img
                        src={post.blogImage}
                        alt={post.blogTitle}
                        className="blog-card-image"
                        whileHover={{ scale: 1.15 }}
                      />
                    </div>

                    <div className="blog-card-body">
                      <h3 className="blog-card-title">{post.blogTitle}</h3>

                      <p
                        className={`blog-card-desc ${
                          isExpanded ? "expanded" : ""
                        }`}
                      >
                        {post.blogDescription}
                      </p>

                      {post.blogDescription?.length > 120 && (
                        <span
                          className="blog-read-more"
                          onClick={(e) => {
                            e.stopPropagation();
                            navigate(`/blog/${post.blogId}`);
                          }}
                        >
                          Read More
                        </span>
                      )}

                      <div className="blog-card-meta">
                        <Text className="blog-card-date">
                          {post.createdAt
                            ? new Date(post.createdAt).toDateString()
                            : ""}
                        </Text>

                        <Tag className="blog-card-tag">Real Estate</Tag>
                      </div>
                    </div>
                  </motion.article>
                </Col>
              </Row>
            </div>
          );
        })}
      </Slider>
    </section>
  );
};

export default BlogSection;
