/* eslint-disable @typescript-eslint/no-explicit-any */
import {
  Count,
  CountSchema,
  Filter,
  FilterExcludingWhere,
  repository,
  Where,
} from '@loopback/repository';
import {
  post,
  param,
  get,
  getModelSchemaRef,
  patch,
  put,
  del,
  requestBody,
  response,
} from '@loopback/rest';
import {LogBuyer} from '../models';
import {LogBuyerRepository} from '../repositories';

// Helper function to remove unwanted fields
function cleanLogBuyerData(data: any): any {
  const fieldsToRemove = [
    'isDefault',
    'isOn',
    'isActive',
    'createdDate',
    'createdBy',
    'modifiedDate',
    'modifiedBy',
    'deletedDate',
    'deletedBy',
    'deleteRemark',
    'lastChanged'
  ];

  // Deep clone to avoid modifying original object
  const cleanData = JSON.parse(JSON.stringify(data));

  // Remove fields from main object
  fieldsToRemove.forEach(field => {
    delete cleanData[field];
  });

  // Remove fields from postProperty if it exists
  if (cleanData.postProperty) {
    fieldsToRemove.forEach(field => {
      delete cleanData.postProperty[field];
    });
  }

  // Remove fields from buyer if it exists
  if (cleanData.buyer) {
    fieldsToRemove.forEach(field => {
      delete cleanData.buyer[field];
    });
  }

  return cleanData;
}

export class LogBuyerController {
  constructor(
    @repository(LogBuyerRepository)
    public logBuyerRepository: LogBuyerRepository,
  ) {}

  @post('/log-buyers')
  @response(200, {
    description: 'LogBuyer model instance',
    content: {'application/json': {schema: getModelSchemaRef(LogBuyer)}},
  })
  async create(
    @requestBody({
      content: {
        'application/json': {
          schema: getModelSchemaRef(LogBuyer, {
            title: 'NewLogBuyer',
            exclude: ['logBuyerId'],
          }),
        },
      },
    })
    logBuyer: Omit<LogBuyer, 'logBuyerId'>,
  ): Promise<LogBuyer> {
    return this.logBuyerRepository.create(logBuyer);
  }

  @get('/log-buyers/count')
  @response(200, {
    description: 'LogBuyer model count',
    content: {'application/json': {schema: CountSchema}},
  })
  async count(
    @param.where(LogBuyer) where?: Where<LogBuyer>,
  ): Promise<Count> {
    return this.logBuyerRepository.count(where);
  }

  @get('/log-buyers')
  @response(200, {
    description: 'Array of LogBuyer model instances',
    content: {
      'application/json': {
        schema: {
          type: 'array',
          items: getModelSchemaRef(LogBuyer, {includeRelations: true}),
        },
      },
    },
  })
  async find(
    @param.filter(LogBuyer) filter?: Filter<LogBuyer>,
  ): Promise<any[]> {
    // Add buyer and postProperty relations to filter
    const f: Filter<LogBuyer> = {
      ...filter,
      include: ['buyer', 'postProperty'],
    };
    const data = await this.logBuyerRepository.find(f);
    return data.map(cleanLogBuyerData);
  }

  @patch('/log-buyers')
  @response(200, {
    description: 'LogBuyer PATCH success count',
    content: {'application/json': {schema: CountSchema}},
  })
  async updateAll(
    @requestBody({
      content: {
        'application/json': {
          schema: getModelSchemaRef(LogBuyer, {partial: true}),
        },
      },
    })
    logBuyer: LogBuyer,
    @param.where(LogBuyer) where?: Where<LogBuyer>,
  ): Promise<Count> {
    return this.logBuyerRepository.updateAll(logBuyer, where);
  }

  @get('/log-buyers/{id}')
  @response(200, {
    description: 'LogBuyer model instance',
    content: {
      'application/json': {
        schema: getModelSchemaRef(LogBuyer, {includeRelations: true}),
      },
    },
  })
  async findById(
    @param.path.number('id') id: number,
    @param.filter(LogBuyer, {exclude: 'where'}) filter?: FilterExcludingWhere<LogBuyer>
  ): Promise<any> {
    // Add buyer and postProperty relations to filter
    const f: FilterExcludingWhere<LogBuyer> = {
      ...filter,
      include: ['buyer', 'postProperty'],
    };
    const data = await this.logBuyerRepository.findById(id, f);
    return cleanLogBuyerData(data);
  }

  @patch('/log-buyers/{id}')
  @response(204, {
    description: 'LogBuyer PATCH success',
  })
  async updateById(
    @param.path.number('id') id: number,
    @requestBody({
      content: {
        'application/json': {
          schema: getModelSchemaRef(LogBuyer, {partial: true}),
        },
      },
    })
    logBuyer: LogBuyer,
  ): Promise<void> {
    await this.logBuyerRepository.updateById(id, logBuyer);
  }

  @put('/log-buyers/{id}')
  @response(204, {
    description: 'LogBuyer PUT success',
  })
  async replaceById(
    @param.path.number('id') id: number,
    @requestBody() logBuyer: LogBuyer,
  ): Promise<void> {
    await this.logBuyerRepository.replaceById(id, logBuyer);
  }

  @del('/log-buyers/{id}')
  @response(204, {
    description: 'LogBuyer DELETE success',
  })
  async deleteById(@param.path.number('id') id: number): Promise<void> {
    await this.logBuyerRepository.deleteById(id);
  }
}
